package ru.yandex.canvas.model.html_builder;

import java.io.IOException;
import java.io.StringWriter;
import java.net.URISyntaxException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.collect.ImmutableMap;

import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.MediaSetSubItem;
import ru.yandex.canvas.model.elements.Image;
import ru.yandex.canvas.model.elements.SecondImage;
import ru.yandex.canvas.model.elements.Video;
import ru.yandex.canvas.model.video.files.InBannerVideo;
import ru.yandex.canvas.model.video.files.StreamFormat;
import ru.yandex.canvas.repository.video.VideoFilesRepository;
import ru.yandex.canvas.service.video.DCParams;
import ru.yandex.canvas.service.video.InBannerVideoFilesService;
import ru.yandex.canvas.service.video.PreviewData;
import ru.yandex.canvas.service.video.VAST;

import static ru.yandex.canvas.VideoConstants.DEFAULT_VPAID_PCODE_URL;
import static ru.yandex.direct.utils.CommonUtils.ifNotNullOrDefault;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class InBannerHtmlCreativeWriter implements HtmlFromCreativeWriter {
    private final static String IN_BANNER_TEMPLATE = "in_banner_rtb_template.mustache";
    private final static String IN_BANNER_IVI_TEMPLATE = "in_banner_ivi_rtb_template.mustache";
    private final static String IN_BANNER_IV_TEMPLATE = "in_banner_iv_rtb_template.mustache";
    private final static String IN_BANNER_VIDEO_ON_LEFT_TEMPLATE = "in_banner_video_on_left_template.mustache";
    private final static String IN_BANNER_VIDEO_ON_RIGHT_TEMPLATE = "in_banner_video_on_right_template.mustache";
    private final static String IN_BANNER_SCREENSHOT_VIDEO_ON_LEFT_TEMPLATE = "in_banner_screenshot_video_on_left_template.mustache";
    private final static String IN_BANNER_SCREENSHOT_VIDEO_ON_RIGHT_TEMPLATE = "in_banner_screenshot_video_on_right_template.mustache";
    private final static String IN_BANNER_VAST_TEMPLATE = "in_banner_vast_template.xml";
    private final static String IN_BANNER_SCREENSHOT_HTML_TEMPLATE = "in_banner_screenshot_html_template.mustache";
    private final static String IN_BANNER_IV_SCREENSHOT_HTML_TEMPLATE = "in_banner_screenshot_html_template_iv" +
            ".mustache";
    private final static String IN_BANNER_IVI_SCREENSHOT_HTML_TEMPLATE = "in_banner_screenshot_html_template_ivi" +
            ".mustache";

    private final static Map<Integer, Mustache> templateByPreset =
            Map.of(21, new DefaultMustacheFactory().compile(IN_BANNER_TEMPLATE),
                    22, new DefaultMustacheFactory().compile(IN_BANNER_IVI_TEMPLATE),
                    23, new DefaultMustacheFactory().compile(IN_BANNER_IVI_TEMPLATE),
                    24, new DefaultMustacheFactory().compile(IN_BANNER_IV_TEMPLATE),
                    25, new DefaultMustacheFactory().compile(IN_BANNER_IVI_TEMPLATE),
                    26, new DefaultMustacheFactory().compile(IN_BANNER_VIDEO_ON_LEFT_TEMPLATE),
                    27, new DefaultMustacheFactory().compile(IN_BANNER_VIDEO_ON_RIGHT_TEMPLATE)
            );


    private final static Map<Integer, Mustache> screenshotTemplateByPreset =
            Map.of(
                    21, new DefaultMustacheFactory().compile(IN_BANNER_SCREENSHOT_HTML_TEMPLATE),
                    22, new DefaultMustacheFactory().compile(IN_BANNER_IVI_SCREENSHOT_HTML_TEMPLATE),
                    23, new DefaultMustacheFactory().compile(IN_BANNER_IVI_SCREENSHOT_HTML_TEMPLATE),
                    24, new DefaultMustacheFactory().compile(IN_BANNER_IV_SCREENSHOT_HTML_TEMPLATE),
                    25, new DefaultMustacheFactory().compile(IN_BANNER_IVI_SCREENSHOT_HTML_TEMPLATE),
                    26, new DefaultMustacheFactory().compile(IN_BANNER_SCREENSHOT_VIDEO_ON_LEFT_TEMPLATE),
                    27, new DefaultMustacheFactory().compile(IN_BANNER_SCREENSHOT_VIDEO_ON_RIGHT_TEMPLATE)
            );

    private final InBannerVideoFilesService inBannerVideoFilesService;

    public InBannerHtmlCreativeWriter(InBannerVideoFilesService inBannerVideoFilesService) {
        this.inBannerVideoFilesService = inBannerVideoFilesService;
    }

    //Есть формат 240х432 который на самом деле не 16х9, и у него из-за этого черная полоса сбоку.
    public static List<StreamFormat> filterBrokenFormats(List<StreamFormat> formats) {
        return formats.stream().filter(
                e -> e.getHeight() == null || 240 != e.getHeight() || e.getWidth() == null || 432 != e.getWidth()
        ).collect(Collectors.toList());
    }

    //todo: backslash any single quotes
    String getVast(CreativeDocument creativeDocument, Long clientId) throws IOException {
        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder();

        if (clientId != null) {
            queryBuilder.withClientId(clientId);
        }

        InBannerVideo inBannerVideo = inBannerVideoFilesService.lookupMovie(creativeDocument, queryBuilder);

        if (inBannerVideo == null) {
            throw new NotFoundException("Video not found");
        }

        VAST vast;

        try {
            vast = VAST.builder()
                    .setCustomTemplateName(IN_BANNER_VAST_TEMPLATE)
                    .setMediaFiles(filterBrokenFormats(inBannerVideo.getFormats()))
                    .setCreativeId(creativeDocument.getId())
                    .setDuration(new TimeDelta(inBannerVideo.getDuration()))
                    .setSkipOffset(new TimeDelta(5))
                    .setAddPixelImpression(false)
                    .setUseVpaidImpressions(false)
                    .setUseTrackingEvents(false)
                    .setVpaidPcodeUrl(DEFAULT_VPAID_PCODE_URL)
                    .setStrmPrefix(inBannerVideo.getStrmPrefix())
                    .setShowVideoClicks(true)
                    .build();
        } catch (IOException | URISyntaxException e) {
            throw new RuntimeException(e);
        }

        return vast.injectDcParams(new DCParams(new PreviewData(), null, false));
    }

    public String composeCreativeHtmlForScreenshot(CreativeDocument creativeDocument) {
        StringWriter writer = new StringWriter();

        Video videoItem = inBannerVideoFilesService.getVideoElement(creativeDocument);
        Image imageItem = inBannerVideoFilesService.getImageElement(creativeDocument);
        SecondImage secondImage = inBannerVideoFilesService.getSecondImageElement(creativeDocument);

        InBannerVideo inBannerVideo = inBannerVideoFilesService.lookupMovie(creativeDocument,
                new VideoFilesRepository.QueryBuilder());

        if (videoItem == null || inBannerVideo == null) {
            throw new NotFoundException("Video not found");
        }

        screenshotTemplateByPreset.get(creativeDocument.getPresetId()).execute(writer,
                ImmutableMap.builder()
                        .put("body_width", creativeDocument.getData().getWidth())
                        .put("body_height", creativeDocument.getData().getHeight())
                        .put("picture_height", imageItem.getOptions().getHeight())
                        .put("picture_width", imageItem.getOptions().getWidth())
                        .put("picture_is_default",
                                inBannerVideoFilesService.getImageMediaItem(creativeDocument).isDefaultValue())
                        .put("second_picture_is_default",
                                ifNotNullOrDefault(inBannerVideoFilesService.getSecondImageMediaItem(creativeDocument), MediaSetSubItem::isDefaultValue, false))
                        .put("picture2_height", ifNotNullOrDefault(secondImage, e -> e.getOptions().getHeight(), 0L))
                        .put("picture2_width", ifNotNullOrDefault(secondImage, e -> e.getOptions().getWidth(), 0L))
                        .put("video_height", videoItem.getOptions().getHeight())
                        .put("video_width", videoItem.getOptions().getWidth())
                        .put("picture", inBannerVideoFilesService.lookupPicture(creativeDocument))
                        .put("picture2", ifNotNullOrDefault(secondImage,
                                e -> inBannerVideoFilesService.lookupSecondPicture(creativeDocument), ""))
                        .put("frame", inBannerVideo.getThumbnail().getPreview().getUrl())
                        .build()
        );

        return writer.toString();

    }

    public String composeCreativeHTML(CreativeDocument creativeDocument, Long clientId, String dcParams,
                                      String nonce) {

        StringWriter writer = new StringWriter();

        Video videoItem = inBannerVideoFilesService.getVideoElement(creativeDocument);
        Image imageItem = inBannerVideoFilesService.getImageElement(creativeDocument);
        SecondImage secondImage = inBannerVideoFilesService.getSecondImageElement(creativeDocument);

        try {
            templateByPreset.get(creativeDocument.getPresetId()).execute(writer,
                    ImmutableMap.builder()
                            .put("nonce", nvl(nonce, ""))
                            .put("AUCTION_DC_PARAMS", dcParams)
                            .put("body_width", creativeDocument.getData().getWidth()) //TODO get them from preset, not
                            // from request
                            .put("body_height", creativeDocument.getData().getHeight())
                            .put("picture_height", imageItem.getOptions().getHeight())
                            .put("picture_width", imageItem.getOptions().getWidth())
                            .put("picture_is_default",
                                    inBannerVideoFilesService.getImageMediaItem(creativeDocument).isDefaultValue())
                            .put("second_picture_is_default",
                                    ifNotNullOrDefault(inBannerVideoFilesService.getSecondImageMediaItem(creativeDocument), MediaSetSubItem::isDefaultValue, false))
                            .put("picture2_height", ifNotNullOrDefault(secondImage, e -> e.getOptions().getHeight(),
                                    0L))
                            .put("picture2_width", ifNotNullOrDefault(secondImage, e -> e.getOptions().getWidth(), 0L))
                            .put("video_height", videoItem.getOptions().getHeight())
                            .put("video_width", videoItem.getOptions().getWidth())
                            .put("picture", inBannerVideoFilesService.lookupPicture(creativeDocument))
                            .put("picture2", ifNotNullOrDefault(secondImage,
                                    e -> inBannerVideoFilesService.lookupSecondPicture(creativeDocument), ""))
                            .put("vast", getVast(creativeDocument, clientId).replaceAll("\\r|\\n", ""))
                            .build()
            );
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

        return writer.toString();
    }

    @Override
    public String composeCreativeHTML(CreativeDocument creativeDocument) {
        return composeCreativeHTML(creativeDocument, null, "${AUCTION_DC_PARAMS}", null);
    }

}
