package ru.yandex.canvas.model.presets;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import javax.annotation.Nonnull;

import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.SessionParams;

import static ru.yandex.canvas.model.direct.Privileges.Permission.ADAPTIVE_CREATIVES;

public class PresetSelectionCriteria {
    private final Set<PresetTag> presetTags;
    private final boolean useAdaptive;

    private PresetSelectionCriteria(boolean useAdaptive) {
        this.useAdaptive = useAdaptive;
        presetTags = new HashSet<>();
    }

    public static PresetSelectionCriteria any(boolean useAdaptive) {
        PresetSelectionCriteria selectionCriteria = new PresetSelectionCriteria(useAdaptive);
        selectionCriteria.getTags().addAll(Arrays.asList(PresetTag.values()));
        return selectionCriteria;
    }

    public static PresetSelectionCriteriaBuilder builder() {
        return new PresetSelectionCriteriaBuilder();
    }

    @Nonnull
    public static PresetSelectionCriteria fromSessionParams(SessionParams sessionParams, AuthService authService) {
        boolean cpmCommon = !sessionParams.sessionIs(SessionParams.SessionTag.CPM_GEOPRODUCT)
                && !sessionParams.sessionIs(SessionParams.SessionTag.CPM_GEO_PIN);

        return builder()
                .withUseAdaptive(authService.checkPermission(ADAPTIVE_CREATIVES))
                .withCpmCommon(cpmCommon) // Временно, пока для обычных креативов тег не
                // передается
                .withCpmGeoproduct(sessionParams.sessionIs(SessionParams.SessionTag.CPM_GEOPRODUCT))
                .withCpmGeoPin(sessionParams.sessionIs(SessionParams.SessionTag.CPM_GEO_PIN))
                .build();
    }

    /**
     * Возвращает набор {@link PresetTag} для удобной фильтрации
     */
    public Set<PresetTag> getTags() {
        return presetTags;
    }

    public boolean useAdaptive() {
        return useAdaptive;
    }

    public static final class PresetSelectionCriteriaBuilder {
        private boolean cpmCommon;

        private boolean cpmGeoproduct;

        private boolean cpmGeoPin;

        private boolean useAdaptive = false;

        private PresetSelectionCriteriaBuilder() {
        }

        public PresetSelectionCriteriaBuilder withCpmCommon(boolean cpmCommon) {
            this.cpmCommon = cpmCommon;
            return this;
        }

        public PresetSelectionCriteriaBuilder withCpmCommon(long cpmCommon) {
            return withCpmCommon(cpmCommon != 0);
        }

        public PresetSelectionCriteriaBuilder withCpmGeoproduct(boolean cpmGeoproduct) {
            this.cpmGeoproduct = cpmGeoproduct;
            return this;
        }

        public PresetSelectionCriteriaBuilder withCpmGeoPin(boolean cpmGeoPin) {
            this.cpmGeoPin = cpmGeoPin;
            return this;
        }

        public PresetSelectionCriteriaBuilder withCpmGeoproduct(long cpmGeoproduct) {
            return withCpmGeoproduct(cpmGeoproduct != 0);
        }

        public PresetSelectionCriteria build() {
            PresetSelectionCriteria selectionCriteria = new PresetSelectionCriteria(useAdaptive);
            Set<PresetTag> presetTags = selectionCriteria.getTags();
            if (cpmCommon) {
                presetTags.add(PresetTag.COMMON);
            }
            if (cpmGeoproduct) {
                presetTags.add(PresetTag.CPM_GEOPRODUCT);
            }
            if (cpmGeoPin) {
                presetTags.add(PresetTag.CPM_GEO_PIN);
            }

            return selectionCriteria;
        }

        public PresetSelectionCriteriaBuilder withUseAdaptive(boolean useAdaptive) {
            this.useAdaptive = useAdaptive;
            return this;
        }
    }
}
