package ru.yandex.canvas.model.validation;

import java.net.IDN;
import java.util.Arrays;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.google.common.base.Utf8;
import com.google.common.net.InternetDomainName;
import org.jetbrains.annotations.NotNull;

import ru.yandex.canvas.model.validation.presetbased.elements.ValidColorValidator;

public class Checkers {
    private static final String SUBTITLES_PREFIX = "WEBVTT\n\n";
    private static final Pattern SUBTITLES_TIMESTAMPS_PATTERN = Pattern.compile(
            "^\\d{2,2}:\\d{2,2}\\.\\d{2,3} +--> +\\d{2,2}:\\d{2,2}\\.\\d{2,3}\\n\\s*[^\\s].*", Pattern.DOTALL);

    public static boolean checkDomain(@NotNull String domain) {
        try {
            String punyCode = IDN.toASCII(domain);
            return punyCode.matches("(?u)^(?:[a-zA-Z0-9\\-]{1,64}\\.)+[a-zA-Z0-9\\-]{2,20}$")
                    && InternetDomainName.isValid(punyCode);
        } catch (IllegalArgumentException e) {
            return false;
        }
    }

    public static boolean checkSubtitles(@NotNull String subtitles) {
        // строка может быть большой, поэтому не делаем одну общую регулярку, а проверяем по блокам,
        // чтобы не получить переполнение стека
        if (!subtitles.startsWith(SUBTITLES_PREFIX)) {
            return false;
        }
        return Arrays.stream(subtitles.substring(SUBTITLES_PREFIX.length()).split("\n\n"))
                .allMatch(block -> SUBTITLES_TIMESTAMPS_PATTERN.matcher(block).matches());
    }

    public static boolean checkLength(@NotNull String s, @NotNull Integer max, @Nullable Integer min) {
        return (max < 0 || s.length() <= max) && (min == null || s.length() >= min);
    }

    public static boolean checkColor(@NotNull String color) {
        return color.matches(ValidColorValidator.COLOR_PATTERN);
    }

    public static boolean checkUnicodeText(@NotNull String text) {
        /* guava-less solution: https://paste.yandex-team.ru/491186 */
        return text.codePoints()
                .map(cp -> Utf8.encodedLength(String.valueOf(Character.toChars(cp))))
                .allMatch(len -> len <= 3);
    }
}
