package ru.yandex.canvas.model.validation;

import java.util.List;
import java.util.Optional;

import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;

import com.google.common.base.Preconditions;
import org.hibernate.validator.constraintvalidation.HibernateConstraintValidatorContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.Errors;
import org.springframework.validation.FieldError;
import org.springframework.validation.ObjectError;

import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.CreativeDocumentBatch;

/**
 * Валидатор для {@link CreativeDocument} и его дочерних полей на основе PresetId
 */
public class CreativeDocumentBatchPresetBasedValidator
        implements ConstraintValidator<PresetBasedValidation, CreativeDocumentBatch> {
    private static final Logger logger = LoggerFactory.getLogger(CreativeDocumentBatchPresetBasedValidator.class);


    @Override
    public void initialize(PresetBasedValidation constraintAnnotation) {
    }

    @Override
    public boolean isValid(CreativeDocumentBatch batch, ConstraintValidatorContext context) {
        Errors errors = new BeanPropertyBindingResult(batch, "");

        List<CreativeDocument> items = batch.getItems();
        for (int itemIdx = 0; itemIdx < items.size(); itemIdx++) {
            CreativeDocument creative = items.get(itemIdx);
            if (creative == null) {
                errors.rejectValue(String.format("items[%d].", itemIdx),
                        "creative.empty", "Creative is empty");
            }
        }

        if (!errors.hasErrors()) {
            return true;
        }

        processErrors(context, errors);
        return false;
    }

    /**
     * Обрабатывает ошибки из переданного параметра {@link Errors} и передает их в контекст.
     */
    private void processErrors(ConstraintValidatorContext context, Errors errors) {
        HibernateConstraintValidatorContext ctx = context.unwrap(HibernateConstraintValidatorContext.class);
        for (ObjectError error : errors.getAllErrors()) {
            processExpressionVariables(ctx, error);

            ConstraintValidatorContext.ConstraintViolationBuilder builder = ctx
                    .buildConstraintViolationWithTemplate(error.getDefaultMessage());

            if (error instanceof FieldError) {
                builder.addNode(((FieldError) error).getField());
            }

            builder.addConstraintViolation();
        }
    }

    /**
     * Вытаскивает из ошибки ее аргументы и формирует из них значения для placeholder'ов в сообщении об ошибке.
     * Аргументы передаются массивом, длина которого должна быть кратна 2.
     * Сначала идет ключ-placeholder, за ним следует значение.
     */
    private void processExpressionVariables(HibernateConstraintValidatorContext ctx, ObjectError error) {
        Object[] arguments = Optional.ofNullable(error.getArguments()).orElse(new Object[0]);
        Preconditions.checkArgument(arguments.length % 2 == 0,
                "Arguments count should be even (key, value)");
        for (int i = 0; i < arguments.length; i += 2) {
            ctx.addMessageParameter(arguments[i].toString(), arguments[i + 1]);
        }
    }
}
