package ru.yandex.canvas.model.validation;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.regex.Pattern;

import javax.validation.Constraint;
import javax.validation.Payload;

import ru.yandex.canvas.model.validation.presetbased.elements.ValidColorValidator;
import ru.yandex.canvas.service.color.CIEDE2000ColorDistance;
import ru.yandex.canvas.service.color.WGAG20Distance;


@Target({ElementType.FIELD, ElementType.TYPE})
@Retention(RetentionPolicy.RUNTIME)
@Constraint(validatedBy = {DifferentlyColoredOptionsValidator.class,
        DifferentlyColoredCreativeOptionsValidator.class})
public @interface DifferentlyColoredOptions {

    double distance();

    Metric metric();

    String message() default "{colors_are_the_same}";

    Class<?>[] groups() default {};

    Class<? extends Payload>[] payload() default {};

    enum Metric {
        WGAG20(WGAG20Distance::compute),
        CIEDE2000(CIEDE2000ColorDistance::compute);

        private static final Pattern COLOR_RE = Pattern.compile(ValidColorValidator.COLOR_PATTERN);
        private BiFunction<String, String, Double> distanceFunction;

        Metric(BiFunction<String, String, Double> distanceFunction) {
            this.distanceFunction = distanceFunction;
        }

        public boolean almostTheSameColors(final String color1, final String color2, final double distance) {
            if (color1 == null || color2 == null) {
                return false;  // nulls are always different
            }
            if (!COLOR_RE.matcher(color1).matches() || !COLOR_RE.matcher(color2).matches()) {
                return false;  // not-a-colors are always different
            }
            return Objects.equals(color1, color2) || distanceFunction.apply(color1, color2) < distance;
        }
    }
}
