package ru.yandex.canvas.model.validation;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import one.util.streamex.StreamEx;

import ru.yandex.canvas.Html5Constants;
import ru.yandex.canvas.model.Size;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.db.PpcPropertyNames;
import ru.yandex.direct.feature.FeatureName;

import static ru.yandex.canvas.Html5Constants.HTML5_PRESETS_BY_ID;
import static ru.yandex.canvas.Html5Constants.VALID_SIZES_MARKET;
import static ru.yandex.canvas.service.SessionParams.SessionTag.CPM_BANNER;
import static ru.yandex.canvas.service.SessionParams.SessionTag.CPM_PRICE;
import static ru.yandex.direct.feature.FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES;
import static ru.yandex.direct.feature.FeatureName.WIDE_MARKET;
import static ru.yandex.direct.utils.CommonUtils.max;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class Html5SizeValidator {
    private final static int DEFAULT_INACCURACY = 3;
    private final static int DEFAULT_MAX_RATIO = 4;
    private final SessionParams sessionParams;
    private final PpcPropertiesSupport ppcPropertiesSupport;

    public Html5SizeValidator(SessionParams sessionParams, PpcPropertiesSupport ppcPropertiesSupport) {
        this.sessionParams = sessionParams;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
    }

    public Set<Size> validSizes(Set<String> features) {

        List<Integer> html5Presets = nvl(sessionParams.getHtml5PresetIds(), List.of());

        if (!html5Presets.isEmpty()) {
            return html5Presets.stream()
                    .filter(HTML5_PRESETS_BY_ID::containsKey)
                    .map(e -> HTML5_PRESETS_BY_ID.get(e).getSize())
                    .collect(Collectors.toSet());
        }

        SessionParams.SessionTag productType = sessionParams.getHtml5SessionTag().getSessionTag();

        var sizes = Html5Constants.PRODUCT_VALID_SIZES.get(productType);

        if (features.contains(WIDE_MARKET.getName()) && productType == CPM_BANNER) {
            sizes = ImmutableSet.<Size>builder()
                    .addAll(sizes)
                    .addAll(VALID_SIZES_MARKET)
                    .build();
        }
        //логика про прайсовые в параметрах запроса приходит
        if (productType == CPM_PRICE && sessionParams.getFilter() != null
                && sessionParams.getFilter().getSizes() != null && !sessionParams.getFilter().getSizes().isEmpty()) {
            Set<Size> filterSizes = sessionParams.getFilter().getSizes()
                    .stream()
                    .map(it -> new Size(it.getW(), it.getH()))
                    .collect(Collectors.toSet());
            sizes = ImmutableSet.<Size>builder()
                    .addAll(sizes.stream().filter(e -> filterSizes.contains(e))
                            .collect(Collectors.toSet()))
                    .build();
        }

        return sizes;
    }

    public boolean isSizesValid(Collection<Size> sizeList, Set<String> features, SessionParams.Html5Tag productType) {
        var validSizes = validSizes(features);
        if (features.contains(FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES.getName())
                 && productType == SessionParams.Html5Tag.HTML5_CPM_BANNER) {
            return StreamEx.of(sizeList)
                    .allMatch(it -> isProportionallyLarger(validSizes, it));
        }
        return StreamEx.of(sizeList)
                .allMatch(it -> validSizes.contains(it) || validSizes.contains(Size.of(0, it.getHeight())));
    }

    public boolean isProportionallyLarger(Set<Size> validSizes, Size size) {
        var baseSize = getBaseSize(validSizes, size);
        return baseSize != null &&
                (validSizes.contains(baseSize) || validSizes.contains(Size.of(0, baseSize.getHeight())));
    }

    private Size getBaseSize(Set<Size> validSize, Size size) {

        var inaccuracy = ppcPropertiesSupport.get(
                PpcPropertyNames.INACCURACY_IN_PERCENTS_FOR_PROPORTIONALLY_LARGER_IMAGES,
                Duration.ofMinutes(5)
        ).getOrDefault(DEFAULT_INACCURACY);
        inaccuracy = getValueInLimits(inaccuracy, 0, 100);

        var maxRatio = ppcPropertiesSupport.get(
                PpcPropertyNames.MAX_RATIO_FOR_PROPORTIONALLY_LARGER_IMAGES,
                Duration.ofMinutes(5)
        ).getOrDefault(DEFAULT_MAX_RATIO);
        maxRatio = getValueInLimits(maxRatio, 1, 100);

        for (var baseSize : validSize) {
            int ratio = (int) Math.round(size.getHeight() * 1d / baseSize.getHeight());
            if (ratio == 0) {
                continue;
            }
            if (ratio <= maxRatio &&
                    Math.abs(baseSize.getHeight() * ratio - size.getHeight()) <= baseSize.getHeight() * inaccuracy.doubleValue() / 100) {
                if (Math.abs(baseSize.getWidth() * ratio - size.getWidth()) <= baseSize.getWidth() * inaccuracy.doubleValue() / 100) {
                    return baseSize;
                }
                if (baseSize.getWidth() == 0) {
                    return new Size(size.getWidth() / ratio, baseSize.getHeight());
                }
            }
        }
        return null;
    }


    public Size getBaseSize(Size size, Set<String> features, SessionParams.Html5Tag productType) {
        if (features.contains(ALLOW_PROPORTIONALLY_LARGER_IMAGES.getName())
                && productType == SessionParams.Html5Tag.HTML5_CPM_BANNER) {
            var validSizes = validSizes(features);
            var baseSize = getBaseSize(validSizes, size);
            return baseSize == null ? size : baseSize;
        }
        return size;
    }

    public String validSizesByProductTypeString(Set<String> features) {
        return StreamEx.of(validSizes(features))
                .map(Object::toString).sorted().joining(", ");
    }

    private static int getValueInLimits(int value, int min, int max) {
        value = Math.max(value, min);
        return Math.min(value, max);
    }
}
