package ru.yandex.canvas.model.validation.presetbased.creative;

import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.springframework.validation.Errors;

import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.elements.Element;
import ru.yandex.canvas.model.elements.ElementType;

import static java.util.function.Function.identity;

public class ElementsAvailabilityValidator extends CreativeDataValidator {

    private static final String EITHER_DOMAIN_OR_PHONE_CAN_BE_FILLED = "{only_one_of_domain_and_phone}";

    @Nullable
    private Boolean eitherDomainOrPhoneCanBeFilled;

    @Nullable
    public Boolean getEitherDomainOrPhoneCanBeFilled() {
        return eitherDomainOrPhoneCanBeFilled;
    }

    public void setEitherDomainOrPhoneCanBeFilled(Boolean eitherDomainOrPhoneCanBeFilled) {
        this.eitherDomainOrPhoneCanBeFilled = eitherDomainOrPhoneCanBeFilled;
    }

    @Override
    public boolean isSupported(CreativeData target) {
        if (target == null) {
            return false;
        }
        return Boolean.TRUE.equals(eitherDomainOrPhoneCanBeFilled);
    }

    @Override
    public void validate(CreativeData creative, @NonNull Errors errors) {
        List<Element> elements = creative.getElements();
        Map<String, Integer> indexByType = EntryStream.of(elements)
                .mapValues(Element::getType)
                .invert()
                .toMap();
        Map<String, Element> elementByType = StreamEx.of(elements).toMap(Element::getType, identity());
        if (Boolean.TRUE.equals(eitherDomainOrPhoneCanBeFilled)) {
            validateDomainAndPhone(elementByType, indexByType, errors);
        }
    }

    private void validateDomainAndPhone(Map<String, Element> elementByType, Map<String, Integer> indexByType,
                                        @NonNull Errors errors) {
        Element domain = elementByType.get(ElementType.DOMAIN);
        Element phone = elementByType.get(ElementType.PHONE);
        if (domain == null ||
                !domain.getAvailable() ||
                phone == null ||
                !phone.getAvailable()) {
            return;
        }
        String domainField = String.format("elements[%d].options.content", indexByType.get(ElementType.DOMAIN));
        errors.rejectValue(domainField, "consistency", EITHER_DOMAIN_OR_PHONE_CAN_BE_FILLED);
        String phoneField = String.format("elements[%d].options.content", indexByType.get(ElementType.PHONE));
        errors.rejectValue(phoneField, "consistency", EITHER_DOMAIN_OR_PHONE_CAN_BE_FILLED);
    }
}
