package ru.yandex.canvas.model.validation.presetbased.creative;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.annotation.Nonnull;

import org.checkerframework.checker.nullness.qual.NonNull;
import org.springframework.validation.Errors;

import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.MediaSet;
import ru.yandex.canvas.model.MediaSetItem;
import ru.yandex.canvas.model.MediaSetSubItem;
import ru.yandex.canvas.model.elements.Element;
import ru.yandex.canvas.model.elements.ElementType;

import static java.util.stream.Collectors.toList;

public abstract class MediaSetValidator<E extends Element> extends CreativeDataValidator {

    @Nonnull
    private List<String> elementTypes = List.of(ElementType.IMAGE, ElementType.SECOND_IMAGE, ElementType.LOGO);

    public MediaSetValidator() {
    }

    public MediaSetValidator(@Nonnull List<String> elementTypes) {
        this.elementTypes = elementTypes;
    }

    public void setElementTypes(@Nonnull List<String> elementTypes) {
        this.elementTypes = elementTypes;
    }

    @Override
    public boolean isSupported(CreativeData target) {
        return target != null && target.getElements() != null && target.getMediaSets() != null;
    }

    @Override
    public void validate(CreativeData target, @NonNull Errors errors) {
        Map<String, MediaSet> mediaSets = target.getMediaSets();
        List<Element> elements = target.getElements();
        for (int i = 0; i < elements.size(); i++) {
            Element element = elements.get(i);
            MediaSet mediaSet = mediaSets.get(element.getMediaSet());
            if (isSupported(element, mediaSet)) {
                try {
                    errors.pushNestedPath(String.format("elements[%d].", i));
                    validate((E) element, mediaSet, target, errors);
                } finally {
                    errors.popNestedPath();
                }
            }
        }
    }

    protected boolean isSupported(Element element, MediaSet mediaSet) {
        return element != null && element.getAvailable() &&
                element.getType() != null && elementTypes.contains(element.getType());
    }

    abstract protected void validate(E element, MediaSet mediaSet, CreativeData creativeData, @NonNull Errors errors);

    protected List<String> getFileIds(MediaSet mediaSet) {
        return getSubItems(mediaSet).stream()
                .map(MediaSetSubItem::getEffectiveFileId)
                .filter(Objects::nonNull)
                .collect(toList());
    }

    protected List<MediaSetSubItem> getSubItems(MediaSet mediaSet) {
        if (mediaSet == null) {
            return Collections.emptyList();
        }

        return mediaSet.getItems().stream()
                .map(MediaSetItem::getItems)
                .flatMap(Collection::stream)
                .filter(e -> !e.isDefaultValue())
                .collect(toList());
    }
}
