package ru.yandex.canvas.model.validation.presetbased.elements;

import javax.annotation.Nonnull;

import com.google.common.base.Utf8;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.springframework.validation.Errors;

import ru.yandex.canvas.model.elements.Element;

/**
 * Validates a {@link String} doesn't contains restricted symbols. (more than 3 utf-8 bytes)
 * @see <a href="https://st.yandex-team.ru/DIRECT-75602">DIRECT-75602</a>
 */
public class ValidUnicodeSymbolsValidator extends ElementValidator<Element> {

    private static final String MESSAGE = "{invalid_symbols}";

    /**
     * Имя поля, которое будет в ошибке валидации.
     * Нужно, например, для ageRestiction, там используется value вместо content
     * @see ru.yandex.canvas.model.elements.AgeRestriction.Options
     */
    @Nonnull
    private String fieldName = "content";

    public ValidUnicodeSymbolsValidator() {
    }

    public ValidUnicodeSymbolsValidator(@Nonnull String fieldName) {
        this.fieldName = fieldName;
    }

    public void setFieldName(@Nonnull String fieldName) {
        this.fieldName = fieldName;
    }

    @Override
    public boolean isSupported(Element element) {
        return element != null && element.getAvailable() && element.getOptions() instanceof Element.ColoredTextOptions;
    }

    @Override
    public void validate(Element element, @NonNull Errors errors) {
        Element.ColoredTextOptions options = ((Element.ColoredTextOptions) element.getOptions());

        if (!isValid(options.getContent())) {
            errors.rejectValue("options." + fieldName, "invalid.symbols", MESSAGE);
        }
    }

    public static boolean isValid(String content) {
        /* guava-less solution: https://paste.yandex-team.ru/491186 */
        return content == null || content.codePoints()
                .map(cp -> Utf8.encodedLength(String.valueOf(Character.toChars(cp))))
                .allMatch(len -> len <= 3);
    }
}
