package ru.yandex.canvas.model.video;

import java.io.IOException;
import java.util.Date;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.springframework.data.annotation.Id;
import org.springframework.data.mongodb.core.index.CompoundIndex;
import org.springframework.data.mongodb.core.index.CompoundIndexes;
import org.springframework.data.mongodb.core.mapping.Document;
import org.springframework.data.mongodb.core.mapping.Field;

import ru.yandex.canvas.controllers.video.DateSerializer;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.model.CreativeWithId;
import ru.yandex.canvas.model.CreativeWithScreenshot;
import ru.yandex.canvas.model.video.addition.AdditionData;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.Options;
import ru.yandex.canvas.model.video.addition.RtbStatus;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;

@Document(collection = "video_additions")
@CompoundIndexes({
        @CompoundIndex(def = "{'client_id': 1, 'archive': 1, 'date': -1}"),
        @CompoundIndex(def = "{'preset_id': 1, 'client_id': 1}"),
        @CompoundIndex(def = "{'client_id': 1, 'stock_creative_id': 1}", unique = true),
})
@JsonInclude(JsonInclude.Include.ALWAYS)
public class Addition implements CreativeWithScreenshot, CreativeWithId {
    @Id
    private String id;

    @Field("preset_id")
    @JsonProperty("preset_id")
    private Long presetId;

    @Field("name")
    private String name;

    @Field("client_id")
    @JsonProperty("client_id")
    private Long clientId;

    @Field("archive")
    private Boolean archive;

    @Field("status_rtb")
    @JsonProperty("status_rtb")
    private RtbStatus statusRtb;//статус отправки в RTB

    @Field("creation_time")
    @JsonSerialize(using = DateSerializer.class)
    private Date creationTime;

    @Field("date")
    @JsonSerialize(using = DateSerializer.class)
    private Date date;

    @Field("data")
    @NotNull
    @Valid
    private AdditionData data;

    @Field("vast")
    private String vast;

    @Field("screenshot_url")
    @JsonProperty("screenshot_url")
    private String screenshotUrl;

    @Field("screenshot_is_done")
    @JsonProperty("screenshot_is_done")
    private Boolean screenshotIsDone;

    @Field("creative_id")
    @JsonProperty("creative_id")
    private Long creativeId;

    @Field("stock_creative_id")
    @JsonProperty("stock_creative_id")
    private Long stockCreativeId;

    @Field("pythia")
    @JsonProperty("pythia")
    private PythiaParams pythia;

    @Field("skip_url")
    @JsonProperty("skip_url")
    private String skipUrl;

    @Field("vpaid_pcode_url")
    @JsonProperty("vpaid_pcode_url")
    private String vpaidPcodeUrl;

    private String previewUrl;

    public String getId() {
        return id;
    }

    public Addition setId(String id) {
        this.id = id;
        return this;
    }

    public Long getPresetId() {
        return presetId;
    }

    public Addition setPresetId(Long presetId) {
        this.presetId = presetId;
        return this;
    }

    public String getName() {
        return name;
    }

    public Addition setName(String name) {
        this.name = name;
        return this;
    }

    public Long getClientId() {
        return clientId;
    }

    public Addition setClientId(Long clientId) {
        this.clientId = clientId;
        return this;
    }

    public Boolean getArchive() {
        return archive;
    }

    public Addition setArchive(Boolean archive) {
        this.archive = archive;
        return this;
    }

    public Date getCreationTime() {
        return creationTime;
    }

    public Addition setCreationTime(Date creationTime) {
        this.creationTime = creationTime;
        return this;
    }

    public RtbStatus getStatusRtb() {
        return statusRtb;
    }

    public Addition setStatusRtb(RtbStatus statusRtb) {
        this.statusRtb = statusRtb;
        return this;
    }

    public Date getDate() {
        return date;
    }

    public Addition setDate(Date date) {
        this.date = date;
        return this;
    }

    public AdditionData getData() {
        return data;
    }

    public Addition setData(AdditionData data) {
        this.data = data;
        return this;
    }

    public String getVast() {
        return vast;
    }

    public Addition setVast(String vast) {
        this.vast = vast;
        return this;
    }

    public String getScreenshotUrl() {
        return screenshotUrl;
    }

    public Addition setScreenshotUrl(String screenshotUrl) {
        this.screenshotUrl = screenshotUrl;
        return this;
    }

    @Override
    public Long fetchCreativeId() {
        return getCreativeId();
    }

    @Override
    public String fetchScreenshotUrl() {
        return getScreenshotUrl();
    }

    @Override
    public Boolean fetchScreenshotIsDone() {
        return getScreenshotIsDone();
    }

    public Boolean getScreenshotIsDone() {
        return screenshotIsDone;
    }

    public Addition setScreenshotIsDone(Boolean screenshotIsDone) {
        this.screenshotIsDone = screenshotIsDone;
        return this;
    }

    public Long getCreativeId() {
        return creativeId;
    }

    public Addition setCreativeId(Long creativeId) {
        this.creativeId = creativeId;
        return this;
    }

    public Long getStockCreativeId() {
        return stockCreativeId;
    }

    public Addition setStockCreativeId(Long stockCreativeId) {
        this.stockCreativeId = stockCreativeId;
        return this;
    }

    public PythiaParams getPythia() {
        return pythia;
    }

    public Addition setPythia(PythiaParams pythia) {
        this.pythia = pythia;
        return this;
    }

    public String getSkipUrl() {
        return skipUrl;
    }

    public Addition setSkipUrl(String skipUrl) {
        this.skipUrl = skipUrl;
        return this;
    }

    public String getVpaidPcodeUrl() {
        return vpaidPcodeUrl;
    }

    public Addition setVpaidPcodeUrl(String vpaidPcodeUrl) {
        this.vpaidPcodeUrl = vpaidPcodeUrl;
        return this;
    }

    public String getPreviewUrl() {
        return previewUrl;
    }

    public Addition setPreviewUrl(String previewUrl) {
        this.previewUrl = previewUrl;
        return this;
    }

    public AdditionElementOptions findFilesOptions() {
        if (this.getData() == null || this.getData().getElements() == null) {
            throw new InternalServerError();
        }

        return this.getData().getElements()
                .stream()
                .filter(e -> e.getType() == AdditionElement.ElementType.ADDITION)
                .map(e -> (AdditionElementOptions) e.getOptions())
                .findFirst()
                .orElseThrow(InternalServerError::new);
    }


    public AdditionElement findElementByType(AdditionElement.ElementType type) {
        return this.getData().getElements().stream()
                .filter(e -> type.equals(e.getType()))
                .findFirst().get();
    }

    public Addition copy() throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        return objectMapper.readValue(objectMapper.writeValueAsString(this), Addition.class);
    }

    public void trimTextOptions() {
        for (AdditionElement e : getData().getElements()) {
            Options options = e.getOptions();
            String text = options.getText();
            if (text != null) {
                options.setText(text.trim());
            }
        }
    }

    /**
     * Определяет, является ли дополнение оверлеем.
     */
    public boolean isOverlayAddition() {
        // В оверлейных креативах есть элемент, в котором указан overlayId
        return getData().getElements().stream()
                .anyMatch(elem -> elem.getOptions().getOverlayId() != null);
    }

}

