package ru.yandex.canvas.model.video.addition;

import java.lang.reflect.InvocationTargetException;

import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonPropertyOrder;
import com.fasterxml.jackson.annotation.JsonSubTypes;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.JsonValue;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.annotation.AccessType;
import org.springframework.data.annotation.PersistenceConstructor;
import org.springframework.data.mongodb.core.mapping.Field;

import ru.yandex.canvas.MongoLowerCaseEnum;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;
import ru.yandex.canvas.model.video.addition.options.AgeElementOptions;
import ru.yandex.canvas.model.video.addition.options.BodyElementOptions;
import ru.yandex.canvas.model.video.addition.options.ButtonElementOptions;
import ru.yandex.canvas.model.video.addition.options.DisclaimerElementOptions;
import ru.yandex.canvas.model.video.addition.options.DomainElementOptions;
import ru.yandex.canvas.model.video.addition.options.LegalElementOptions;
import ru.yandex.canvas.model.video.addition.options.SubtitlesElementOptions;
import ru.yandex.canvas.model.video.addition.options.TitleElementOptions;

import static org.springframework.data.annotation.AccessType.Type.PROPERTY;

@JsonInclude(JsonInclude.Include.ALWAYS)
@JsonPropertyOrder({"type"})
public class AdditionElement {

    //TODO what if we got unknown type, what then?
    public enum ElementType implements MongoLowerCaseEnum {
        BUTTON(ButtonElementOptions.class, "BUTTON"),
        TITLE(TitleElementOptions.class, "TITLE"),
        BODY(BodyElementOptions.class, "BODY"),
        AGE(AgeElementOptions.class, "AGE"),
        DOMAIN(DomainElementOptions.class, "DOMAIN"),
        LEGAL(LegalElementOptions.class, "LEGAL"),
        ADDITION(AdditionElementOptions.class, null),
        DISCLAIMER(DisclaimerElementOptions.class, "DISCLAIMER"),
        SUBTITLES(SubtitlesElementOptions.class, "SUBTITLES");

        private Class<? extends Options> clazz;
        private String pcodeName;

        ElementType(Class<? extends Options> clazz, String pcodeName) {
            this.clazz = clazz;
            this.pcodeName = pcodeName;
        }

        public String getPcodeName() {
            return pcodeName;
        }

        public Class<? extends Options> getElementClass() {
            return clazz;
        }

        @JsonCreator
        public static ElementType forValue(String value) {
            return ElementType.valueOf(value.toUpperCase());
        }

        @JsonValue
        public String toValue() {
            return StringUtils.lowerCase(this.name());
        }
    }

    @NotNull
    @Field("options")
    @AccessType(PROPERTY)
    @JsonTypeInfo(use = JsonTypeInfo.Id.NAME, include = JsonTypeInfo.As.EXTERNAL_PROPERTY, property = "type",
            visible = true)
    @JsonSubTypes({
            @JsonSubTypes.Type(value = ButtonElementOptions.class, name = "button"),
            @JsonSubTypes.Type(value = AdditionElementOptions.class, name = "addition"),
            @JsonSubTypes.Type(value = AgeElementOptions.class, name = "age"),
            @JsonSubTypes.Type(value = BodyElementOptions.class, name = "body"),
            @JsonSubTypes.Type(value = DisclaimerElementOptions.class, name = "disclaimer"),
            @JsonSubTypes.Type(value = DomainElementOptions.class, name = "domain"),
            @JsonSubTypes.Type(value = LegalElementOptions.class, name = "legal"),
            @JsonSubTypes.Type(value = TitleElementOptions.class, name = "title"),
            @JsonSubTypes.Type(value = SubtitlesElementOptions.class, name = "subtitles"),
    })
    private Options options;

    @Field("type")
    @NotNull
    private ElementType type;

    @Field("available")
    private Boolean available;

    public Options getOptions() {
        return options;
    }

    @AccessType(PROPERTY)
    public void setOptions(Options options)
            throws NoSuchMethodException, IllegalAccessException, InvocationTargetException, InstantiationException {
        this.options = type.getElementClass().getConstructor(Options.class).newInstance(options);
    }

    public AdditionElement withOptions(Options options) {
        this.options = options;
        return this;
    }

    public ElementType getType() {
        return type;
    }

    public AdditionElement withType(ElementType type) {
        this.type = type;
        return this;
    }

    public Boolean getAvailable() {
        return available != null ? available : false;
    }

    public AdditionElement withAvailable(Boolean available) {
        this.available = available;
        return this;
    }

    @JsonCreator
    public AdditionElement(@JsonProperty("type") ElementType type, @JsonProperty("options") Options options,
                           @JsonProperty("available") Boolean available) {
        this.type = type;
        this.options = options;
        this.available = available;
    }

    @PersistenceConstructor
    public AdditionElement(ElementType type) {
        this.type = type;
    }

    public AdditionElement() {
        // needed from json unmarshalling
    }
}
