package ru.yandex.canvas.model.video.files;

import java.util.Date;
import java.util.Objects;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.video.AudioFiles;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.service.video.files.MergedFileRecord;

public class AudioSource implements MediaDataSource {
    private String id;
    private Long duration;
    private String name;
    private String stillageId;
    private String stillageUrl;
    private String stockId;
    private Date date;
    private Long clientId;
    private boolean converting = false;

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        AudioSource that = (AudioSource) o;
        return Objects.equals(id, that.id);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id);
    }

    public AudioSource(VideoFiles videoFiles) {
        StillageFileInfo stillageFileInfo = videoFiles.getStillageFileInfo();
        stillageId = stillageFileInfo == null ? null : stillageFileInfo.getId();
        stillageUrl = stillageFileInfo == null ? null : stillageFileInfo.getUrl();
        id = videoFiles.getId();
        duration = Math.round(videoFiles.getDuration());
        name = videoFiles.getName();
    }

    public AudioSource(MergedFileRecord.AudioDescription audioDescription) {
        id = audioDescription.getStillageId();
        name = audioDescription.getName();
        duration = audioDescription.getDuration();
        stillageId = audioDescription.getStillageId();
        stillageUrl = audioDescription.getStillageUrl();
        stockId = audioDescription.getStillageId();
    }

    public AudioSource(AudioFiles audioFiles) {
        stillageId = audioFiles.getStillageId();
        stillageUrl = audioFiles.getUrl();
        id = audioFiles.getId();
        duration = Math.round(audioFiles.getDuration());
        name = audioFiles.getName();
        converting = audioFiles.getStatus() != null && audioFiles.getStatus() == FileStatus.CONVERTING;
    }

    public AudioSource(MergedFileRecord root) {
        MergedFileRecord.AudioDescription description = root.getAudio();

        id = getIdFromDescription(root);
        name = description.getName();
        duration = description.getDuration();
        stillageId = description.getStillageId();
        stillageUrl = description.getStillageUrl();
        date = null;
        stockId = id;
        clientId = null;
    }

    public AudioSource(VideoFiles dbRef, AudioSource stockAudio) {
        if (dbRef.getStockFileId() == null || dbRef.getType() != FileType.AUDIO) {
            throw new IllegalArgumentException("Cannot make AudioSource object from non-stock object!");
        }

        id = dbRef.getId();
        name = dbRef.getName();
        duration = stockAudio.getDuration();
        stillageId = stockAudio.getStillageId();
        stillageUrl = stockAudio.getStillageUrl();
        stockId = stockAudio.getId();
        date = dbRef.getDate();
        clientId = dbRef.getClientId();
    }

    public static String getIdFromDescription(MergedFileRecord rootRecord) {
        return rootRecord.getAudio().getStillageId();
    }

    public String getId() {
        return id;
    }

    public String getStockId() {
        return stockId;
    }

    public Long getDuration() {
        return duration;
    }

    public AudioSource setDuration(Long duration) {
        this.duration = duration;
        return this;
    }

    public String getName() {
        return name;
    }

    @Override
    public FileType getSourceType() {
        return FileType.AUDIO;
    }

    public AudioSource setName(String name) {
        this.name = name;
        return this;
    }

    public String getStillageId() {
        return stillageId;
    }

    public AudioSource setStillageId(String stillageId) {
        this.stillageId = stillageId;
        return this;
    }

    public String getStillageUrl() {
        return stillageUrl;
    }

    public AudioSource setStillageUrl(String stillageUrl) {
        this.stillageUrl = stillageUrl;
        return this;
    }

    public Date getDate() {
        return date;
    }

    public Long getClientId() {
        return clientId;
    }

    public boolean isNotReady() {
        return converting;
    }

    @Override
    public String toString() {
        return "AudioSource{" +
                "id='" + id + '\'' +
                ", duration=" + duration +
                ", name='" + name + '\'' +
                ", stillageId='" + stillageId + '\'' +
                ", stillageUrl='" + stillageUrl + '\'' +
                ", stockId='" + stockId + '\'' +
                ", date=" + date +
                ", clientId=" + clientId +
                '}';
    }
}
