package ru.yandex.canvas.model.video.files;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.canvas.model.stillage.StillageInfoConverter;
import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.service.video.Ratio;
import ru.yandex.canvas.service.video.VideoMetaData;
import ru.yandex.canvas.service.video.files.MergedFileRecord;

import static ru.yandex.canvas.model.video.files.VideoSource.getIdFromDescription;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

public class MovieAndVideoSourceFactory {
    private final StillageInfoConverter stillageInfoConverter;

    public MovieAndVideoSourceFactory(StillageInfoConverter stillageInfoConverter) {
        this.stillageInfoConverter = stillageInfoConverter;
    }

    public Movie movieFromVideoAudioSources(MergedFileRecord record, VideoSource videoSource, AudioSource audioSource) {
        List<StreamFormat> formats = record.getFormatsList().stream().map(StreamFormat::new)
                .collect(Collectors.toList());

        return new Movie(videoSource, audioSource, formats, record.getStrmPrefix(), record.getStillageUrl(),
                null, true, FileStatus.READY, 15.0, false);
    }

    public Movie movieFromVideoFile(VideoFiles videoFiles) {
        if (videoFiles.getStockFileId() != null || videoFiles.getType() != FileType.VIDEO) {
            throw new IllegalArgumentException("Cannot make videoStream object from stock VideoFiles object!");
        }
        List<StreamFormat> formats = videoFiles.getFormats() == null ? Collections.emptyList()
                : videoFiles.getFormats().stream()
                .filter(it -> !nvl(it.getFormatName(), "").equals("EKaltura"))
                .map(StreamFormat::new)
                .collect(Collectors.toList());
        var stillageUrl = videoFiles.getStillageFileInfo() == null ? null : videoFiles.getStillageFileInfo().getUrl();
        var duration = videoFiles.getDuration() == null ? 15.0 : videoFiles.getDuration();

        var movie = new Movie(videoSourcefromVideoFile(videoFiles), null, formats, videoFiles.getStrmPrefix(),
                stillageUrl, videoFiles.getConvertionTaskId(), false, videoFiles.getStatus(), duration,
                videoFiles.getArchive());
        //калтурные стримы
        movie.setEstKaltura(getEKalturaStreamUrl(videoFiles.getFormats(), false))
                .setEstMobileVertical(getEKalturaStreamUrl(videoFiles.getFormats(), true));

        return movie;
    }

    private String getEKalturaStreamUrl(List<VideoFiles.VideoFormat> formats, boolean mobileVertical) {
        if (formats == null)
            return null;
        return formats.stream()
                .filter(it -> nvl(it.getFormatName(), "").equals("EKaltura"))
                .filter(it -> mobileVertical && estMobileVerticalTag(it.getTagsStr())
                        || !mobileVertical && !estMobileVerticalTag(it.getTagsStr()))
                .map(VideoFiles.VideoFormat::getUrl)
                .filter(it -> it != null)
                .map(url -> url.replace("s3://","https://strm.yandex.ru/vod/") + "/-/manifest.mpd")
                .findFirst().orElse(null);
    }

    private static boolean estMobileVerticalTag(List<String> tagsStr){
        return tagsStr != null && tagsStr.contains("EST_MOBILE_VERTICAL");
    }

    /*
       Make new stock movie object based on user defined record
     */
    public Movie movieFromStockMovieAndDbRefs(VideoFiles dbRef, VideoFiles audioRef, Movie stockMovie) {
        if (!stockMovie.isStock()) {
            throw new IllegalArgumentException("Try to create stock movie from non-stock object");
        }

        Boolean archive = false;
        VideoSource videoSource;
        AudioSource audioSource;
        if (dbRef == null) {
            videoSource = stockMovie.getVideoSource();
        } else {
            videoSource = videoSourceFromStockMovie(dbRef, stockMovie.getVideoSource());
            archive = dbRef.getArchive();
        }

        if (audioRef == null) {
            audioSource = stockMovie.getAudioSource();
        } else {
            audioSource = new AudioSource(audioRef, stockMovie.getAudioSource());
            archive = archive || audioRef.getArchive();
        }
        var duration = stockMovie.getDuration() == null ? 15.0 : stockMovie.getDuration();

        return new Movie(videoSource, audioSource, stockMovie.getFormats(), stockMovie.getStrmPrefix(),
                stockMovie.getStillageUrl(), stockMovie.getConvertionTaskId(), true, stockMovie.getStatus(),
                duration, archive);
    }

    public VideoSource videoSourceFromStockMovie(VideoFiles dbRef, VideoSource stockMovie) {
        if (dbRef.getStockFileId() == null || dbRef.getType() != FileType.VIDEO) {
            throw new IllegalArgumentException("Cannot make videoStream object from non-stock VideoFiles object!");
        }

        return new VideoSource.VideoSourceBuilder()
                .setCategoryId(stockMovie.getCategoryId() + "")
                .setThumbnail(stockMovie.getThumbnail())
                .setOverlayColor(stockMovie.getOverlayColor())
                .setStillageId(stockMovie.getStillageId())
                .setName(dbRef.getName())
                .setId(dbRef.getId())
                .setStillageUrl(stockMovie.getStillageUrl())
                .setRatio(nvl(dbRef.getRatio(), "16:9"))
                .setStockId(stockMovie.getId())
                .setDate(dbRef.getDate())
                .setClientId(dbRef.getClientId())
                .build();
    }

    public VideoSource videoSourceFromMergedRecord(MergedFileRecord rootRecord, String name) {
        MergedFileRecord.VideoDescription description = rootRecord.getVideo();
        String ratio = "16:9";
        Integer ratioPercent = null;
        Ratio ratioFromFormats = ratioFromFormats(rootRecord.getFormatsList());
        if (ratioFromFormats != null) {
            ratioPercent = ratioFromFormats.ratioPercent();
            ratio = ratioFromFormats.toString();
        }
        String id = getIdFromDescription(rootRecord);

        return new VideoSource.VideoSourceBuilder()
                .setCategoryId(description.getCategoryId() + "")
                .setThumbnail(new Thumbnail(rootRecord.getThumbnail()))
                .setOverlayColor(description.getOverlayColor())
                .setStillageId(description.getStillageId())
                .setName(name)
                .setId(id)
                .setStillageUrl(description.getStillageUrl())
                .setRatio(ratio)
                .setRatioPercent(ratioPercent)
                .setStockId(id)
                .build();
    }

    private static Ratio ratioFromFormats(List<MergedFileRecord.FormatDescription> formats) {
        if (formats == null) {
            return null;
        }
        return StreamEx.of(formats)
                .filter(it -> it.getHeight() != null && it.getWidth() != null)
                .findAny()
                .map(it -> new Ratio(Integer.parseInt(it.getWidth()), Integer.parseInt(it.getHeight())))
                .orElse(null);
    }

    public VideoSource videoSourcefromVideoFile(VideoFiles record) {
        var thumbnail = record.getThumbnail() == null ? null : new Thumbnail(record.getThumbnail());
        var stillageUrl = record.getStillageFileInfo() == null ? null : record.getStillageFileInfo().getUrl();
        var ratio = record.getRatio() == null ? "16:9" : record.getRatio();
        var mimeType = record.getStillageFileInfo() == null ? null : record.getStillageFileInfo().getMimeType();
        Integer width = null;
        Integer height = null;
        if (record.getStillageFileInfo() != null && record.getStillageFileInfo().getMetadataInfo() != null) {
            List<VideoMetaData.VideoStreamInfo> videoStreamInfos = ifNotNull(
                    stillageInfoConverter.toVideoMetaData(record.getStillageFileInfo()),
                    VideoMetaData::getVideoStreams);
            if (videoStreamInfos != null) {
                width = videoStreamInfos.stream().findFirst()
                        .map(VideoMetaData.VideoStreamInfo::getWidth).orElse(null);
                height = videoStreamInfos.stream().findFirst()
                        .map(VideoMetaData.VideoStreamInfo::getHeight).orElse(null);
            }
        }
        return new VideoSource.VideoSourceBuilder()
                .setThumbnail(thumbnail)
                .setOverlayColor(record.getOverlayColor())
                .setStillageId(record.getStillageId() + "")
                .setVideoMetaId(record.getVideoMetaId())
                .setPlayerId(record.getPlayerId())
                .setFirstFrameUrl(record.getFirstFrameUrl())
                .setSignaturesUrl(record.getSignaturesUrl())
                .setName(record.getName())
                .setId(record.getId())
                .setStillageUrl(stillageUrl)
                .setRatio(ratio)
                .setDate(record.getDate())
                .setMimeType(mimeType)
                .setWidth(width)
                .setHeight(height)
                .build();
    }
}
