package ru.yandex.canvas.model.video.vc.feed;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.nio.ByteBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.univocity.parsers.common.processor.RowListProcessor;
import com.univocity.parsers.csv.CsvFormat;
import com.univocity.parsers.csv.CsvParser;
import com.univocity.parsers.csv.CsvParserSettings;
import com.univocity.parsers.csv.CsvWriter;
import com.univocity.parsers.csv.CsvWriterSettings;
import org.apache.commons.lang3.ArrayUtils;

import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.ValidationErrorsException;

public class VideoConstructorFeedParsed {
    private final List<String[]> rows;
    private String[] fieldNames;
    private final int rowsCount;
    private int fieldsCount;
    private final Map<String, VideoConstructorFeedFieldType> fieldNameToType = new HashMap<>();
    private final Map<String, Integer> fieldNameToIdx = new HashMap<>();
    private final CsvFormat csvFormat;

    public VideoConstructorFeedParsed(byte[] feedContent) {
        // проверяем, что файл в кодировке utf8, иначе выкидываем исключение
        // приравнием это к невозможности парсинга библиотекой, так как там кодировка не проверяется
        CharsetDecoder utf8Decoder = StandardCharsets.UTF_8.newDecoder();
        try {
            utf8Decoder.decode(ByteBuffer.wrap(feedContent));
        } catch (CharacterCodingException e) {
            throw new ValidationErrorsException(List.of("Feed content is not valid utf8."));
        }

        CsvParserSettings csvParserSettings = new CsvParserSettings();
        csvParserSettings.detectFormatAutomatically();
        csvParserSettings.setHeaderExtractionEnabled(true);

        RowListProcessor rowProcessor = new RowListProcessor();
        csvParserSettings.setProcessor(rowProcessor);

        CsvParser csvParser = new CsvParser(csvParserSettings);
        csvParser.parse(new ByteArrayInputStream(feedContent), "utf8");

        // сохраняем исходный формат для возможной последующей записи (разделители и тд)
        this.csvFormat = csvParser.getDetectedFormat();

        this.rows = rowProcessor.getRows();
        this.fieldNames = rowProcessor.getHeaders();

        this.rowsCount = rows == null ? 0 : rows.size();
        this.fieldsCount = fieldNames == null ? 0 : fieldNames.length;

        if (this.fieldNames != null) {
            IntStream.range(0, this.fieldNames.length).forEach(idx -> this.fieldNameToIdx.put(this.fieldNames[idx], idx));
        }
    }

    /*
    Получить количество строк (первая строка фида с колонками не считается)
     */
    public int getRowsCount() {
        return rowsCount;
    }

    /*
    Получить количество полей
     */
    public int getFieldsCount() {
        return fieldsCount;
    }

    /*
    Получить названия полей
     */
    public String[] getFieldNames() {
        return fieldNames;
    }

    /*
    Получить конкретную строку по ее номеру
     */
    public String[] getRowValues(int idx) {
        if (idx < 0 || idx >= rowsCount) {
            throw new BadRequestException(String.format("Row index %d is out of bounds", idx));
        }
        return rows.get(idx);
    }

    /*
    Получить конкретный столбец по его названию
     */
    public List<String> getFieldValues(String fieldName) {
        checkFieldNameOrThrow(fieldName);
        return IntStream.range(0, rowsCount)
                .mapToObj(i -> getCellValue(i, fieldName))
                .collect(Collectors.toList());
    }

    /*
    Получить элемент в конкретной ячейке
     */
    public String getCellValue(int idx, String fieldName) {
        return getRowValues(idx)[getFieldIdx(fieldName)];
    }

    /*
    Получить порядковый номер поля по его названию
     */
    public int getFieldIdx(String fieldName) {
        checkFieldNameOrThrow(fieldName);
        return fieldNameToIdx.get(fieldName);
    }

    /*
    Объединить значения одного объекта с информацией о полях
     */
    public VideoConstructorFeedRow getRow(int idx) {
        String[] row = getRowValues(idx);
        List<VideoConstructorFeedField> fields = getFields();

        return new VideoConstructorFeedRow(
                IntStream.range(0, fieldsCount)
                        .mapToObj(i -> new VideoConstructorFeedRowItem(row[i], fields.get(i)))
                        .collect(Collectors.toList()));
    }

    /*
    Выставить тип поля
     */
    public void setFieldType(String fieldName, VideoConstructorFeedFieldType fieldType) {
        fieldNameToType.put(fieldName, fieldType);
    }

    /*
    Получить информацию о полях
     */
    public List<VideoConstructorFeedField> getFields() {
        if (fieldsCount == 0) {
            return Collections.emptyList();
        }

        return Arrays.stream(fieldNames)
                .map(name -> new VideoConstructorFeedField()
                        .withName(name)
                        .withType(fieldNameToType.getOrDefault(name, VideoConstructorFeedFieldType.TEXT)))
                .collect(Collectors.toList());
    }

    /*
    Добавить новое поле в конец фида
     */
    public void addNewField(String fieldName, List<String> fieldValues) {
        fieldNames = ArrayUtils.add(fieldNames, fieldName);
        for (int i = 0; i < rowsCount; ++i) {
            rows.set(i, ArrayUtils.add(rows.get(i), fieldValues.get(i)));
        }
        fieldNameToIdx.put(fieldName, fieldsCount);
        ++fieldsCount;
    }

    /*
    Отправить фид на вывод
     */
    public ByteArrayOutputStream write() {
        ByteArrayOutputStream csvResult = new ByteArrayOutputStream();

        CsvWriterSettings csvWriterSettings = new CsvWriterSettings();
        csvWriterSettings.setFormat(csvFormat);

        CsvWriter writer = new CsvWriter(csvResult, csvWriterSettings);
        writer.writeHeaders(fieldNames);
        writer.writeRowsAndClose(rows.toArray(new String[][] {}));

        return csvResult;
    }

    /*
    Проверить присутствие поля в фиде
     */
    public boolean isFieldExists(String fieldName) {
        return fieldNameToIdx.containsKey(fieldName);
    }

    /*
    Проверить присутствие поля в фиде, и если нет, то выкинуть исключение
     */
    private void checkFieldNameOrThrow(String fieldName) {
        if (!isFieldExists(fieldName)) {
            throw new IllegalArgumentException(String.format("Unknown field '%s'", fieldName));
        }
    }
}
