package ru.yandex.canvas.repository.html5;

import java.util.List;

import com.mongodb.client.result.UpdateResult;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.repository.mongo.MongoOperationsWrapper;
import ru.yandex.canvas.repository.mongo.QueryBuilder;

import static java.util.Arrays.asList;

public class SourcesRepository {

    private final MongoOperationsWrapper mongoOperations;

    public SourcesRepository(MongoOperations mongoOperations) {
        this.mongoOperations = new MongoOperationsWrapper(mongoOperations, "canvas_html5_sources");
    }

    public void insertSource(Source source) {
        mongoOperations.insert(source);
    }

    public Source getSourceById(String id, long clientId) {
        return mongoOperations.findOne(
                QueryBuilder.builder()
                        .and(Source.ID.is(id))
                        .and(Source.CLIENT_ID.is(clientId))
                        .and(Source.ARCHIVE.is(false))
                        .build(),
                Source.class);
    }

    public List<Source> getSourcesByIds(long clientId, String... ids) {
        return mongoOperations.find(
                QueryBuilder.builder()
                        .and(Source.ID.in(asList(ids)))
                        .and(Source.CLIENT_ID.is(clientId))
                        .and(Source.ARCHIVE.is(false))
                        .build(),
                Source.class);
    }

    public List<Source> getSourcesWithReplacements() {
        return mongoOperations.find(
                new Query(Criteria.where("html_replacements.0").exists(true)),
                Source.class
        );
    }

    public List<Source> getSourcesWithReplacementsByClientId(Long clientId) {
        return mongoOperations.find(
                new Query(Criteria.where("html_replacements.0").exists(true).and(Source.CLIENT_ID_FIELD_NAME).is(clientId)),
                Source.class
        );
    }

    public UpdateResult archiveSourceById(String id, long clientId) {
        return mongoOperations.updateMulti(
                QueryBuilder.builder()
                        .and(Source.ID.is(id))
                        .and(Source.CLIENT_ID.is(clientId))
                        .and(Source.ARCHIVE.is(false))
                        .build(),
                Update.update(Source.ARCHIVE.getFieldPath(), true),
                Source.class);
    }

    public UpdateResult updateScreenshotUrl(final String id, final String screenshotURL, final Boolean screenshotIsDone) {
        return mongoOperations.updateFirst(
                QueryBuilder.builder()
                        .and(Source.ID.is(id))
                        .and(Source.ARCHIVE.is(false))
                        .build(),
                new Update()
                        .set(Source.SCREENSHOT_URL.getFieldPath(), screenshotURL)
                        .set(Source.SCREENSHOT_IS_DONE.getFieldPath(), screenshotIsDone),
                Source.class);
    }
}
