package ru.yandex.canvas.repository.video;

import java.util.List;
import java.util.Map;
import java.util.function.Supplier;

import com.mongodb.client.result.UpdateResult;
import org.bson.Document;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import ru.yandex.canvas.model.video.AudioFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.video.VideoCreativeType;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class AudioFilesRepository {
    private MongoOperations mongoOperations;
    private DateTimeService dateTimeService;

    public AudioFilesRepository(MongoOperations mongoOperations, DateTimeService dateTimeService) {
        this.mongoOperations = mongoOperations;
        this.dateTimeService = dateTimeService;
    }

    protected <REC> REC databaseWrapper(Supplier<REC> wrapper, String opName) {
        try (TraceProfile profile = Trace.current().profile("db:" + opName, "canvas_audio_files")) {
            return wrapper.get();
        }
    }

    public boolean deleteFile(String fileId, Long clientId) {
        return databaseWrapper(() -> mongoOperations.updateFirst(
                new QueryBuilder().withId(fileId).withClientId(clientId).build(),
                Update.update("archive", true),
                AudioFiles.class)
                  .getModifiedCount(), "update") > 0;
    }

    public AudioFiles save(AudioFiles record) {
        Query query = new QueryBuilder()
                .withClientId(record.getClientId())
                .withName(record.getName())
                .withStillageId(record.getStillageId())
                .build();

        Update update = new Update();
        Document document = new Document();
        mongoOperations.getConverter().write(record, document);
        for (Map.Entry<String, Object> field : document.entrySet()) {
            update.set(field.getKey(), field.getValue());
        }

        return databaseWrapper(() -> upsertAndGet(query, update, record.getClientId()), "save");
    }

    private AudioFiles upsertAndGet(Query query, Update update, Long clientId) {
        UpdateResult result = mongoOperations.upsert(query, update, AudioFiles.class);

        if (result.getModifiedCount() > 0) {
            return mongoOperations.findOne(query, AudioFiles.class);
        } else {
            String id = result.getUpsertedId().asObjectId().getValue().toString();
            return findById(id, clientId);
        }
    }

    public AudioFiles findById(String audioId, Long clientId) {
        return databaseWrapper(() -> mongoOperations.findOne(
                new QueryBuilder().withId(audioId).withClientId(clientId).build(),
                AudioFiles.class), "find");
    }

    public void markFileUsed(String id, Long clientId) {
        Update update = Update.update("archive", false).set("date", dateTimeService.getCurrentDate());
        Query q = new QueryBuilder().withClientId(clientId).withId(id).build();
        databaseWrapper(() -> mongoOperations.upsert(q, update, AudioFiles.class), "upsert");
    }

    //обновить статус и url
    public AudioFiles updateConvertingFile(String id, String url) {
        Update update = Update.update("archive", false).set("date", dateTimeService.getCurrentDate())
                .set("url", url)
                .set("status", FileStatus.READY);
        Query q = new QueryBuilder().withId(id).build();
        databaseWrapper(() -> mongoOperations.upsert(q, update, AudioFiles.class), "upsert");
        return mongoOperations.findOne(q, AudioFiles.class);
    }

    public long count(QueryBuilder queryBuilder) {
        return databaseWrapper(() -> mongoOperations.count(queryBuilder.build(), AudioFiles.class), "count");
    }

    public List<AudioFiles> findByQuery(QueryBuilder queryBuilder, Sort.Direction sortDirection, int limit,
                                        int offset) {
        Query query = queryBuilder.build();
        query.limit(limit);
        query.skip(offset);

        if (sortDirection != null) {
            Sort sort = Sort.by(sortDirection, "date");
            query.with(sort);
        }

        return databaseWrapper(() -> mongoOperations.find(query, AudioFiles.class), "find");
    }

    public static class QueryBuilder extends QueryBuilderBase<QueryBuilder> {

        public QueryBuilder withId(String id) {
            criteries.add(Criteria.where("_id").is(id));
            return this;
        }

        public QueryBuilder withClientId(Long clientId) {
            criteries.add(Criteria.where("client_id").is(clientId));
            return this;
        }

        public QueryBuilder withName(String name) {
            criteries.add(Criteria.where("name").is(name));
            return this;
        }

        public QueryBuilder withStillageId(String stillageId) {
            criteries.add(Criteria.where("stillage_id").is(stillageId));
            return this;
        }

        public QueryBuilder withArchive(Boolean status) {
            criteries.add(Criteria.where("archive").is(status));
            return this;
        }

        public QueryBuilder withNameRegexp(String regexp, String opts) {
            criteries.add(Criteria.where("name").regex(regexp, opts));
            return this;
        }

        public QueryBuilder withCreativeType(VideoCreativeType creativeType) {
            criteries.add(Criteria.where("creativeType").is(creativeType));
            return this;
        }

        public QueryBuilder withCreativeTypeNe(VideoCreativeType creativeType) {
            criteries.add(Criteria.where("creativeType").ne(creativeType));
            return this;
        }

        public QueryBuilder withDurationBetween(double min, double max) {
            criteries.add(Criteria.where("duration").gte(min).lte(max));
            return this;
        }

        @Override
        public Query build() {
            Query query = new Query();

            for (Criteria criteria : criteries) {
                query.addCriteria(criteria);
            }

            return query;
        }
    }
}
