package ru.yandex.canvas.repository.video;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;

import com.mongodb.client.result.UpdateResult;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.data.util.CloseableIterator;

import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.addition.RtbStatus;
import ru.yandex.canvas.repository.ItemsWithTotal;
import ru.yandex.canvas.repository.RepositoryUtils;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class VideoAdditionsRepository {
    private MongoOperations mongoOperations;

    public VideoAdditionsRepository(MongoOperations mongoOperations) {
        this.mongoOperations = mongoOperations;
    }

    public static class QueryBuilder extends QueryBuilderBase<QueryBuilder> {

        public QueryBuilder withId(String id) {
            criteries.add(Criteria.where("_id").is(id));
            return this;
        }

        public QueryBuilder withIds(List<String> ids) {
            criteries.add(Criteria.where("_id").in(ids));
            return this;
        }

        public QueryBuilder withStockCreativeId(Long creativeId) {
            criteries.add(Criteria.where("stock_creative_id").is(creativeId));
            return this;
        }

        public QueryBuilder withStockCreativeIds(List<Long> creativeIds) {
            criteries.add(Criteria.where("stock_creative_id").in(creativeIds));
            return this;
        }

        public QueryBuilder withCreativeId(Long creativeId) {
            criteries.add(Criteria.where("creative_id").in(creativeId));
            return this;
        }

        public QueryBuilder withCreativeIds(List<Long> creativeIds) {
            criteries.add(Criteria.where("creative_id").in(creativeIds));
            return this;
        }

        public QueryBuilder withClientId(Long clientId) {
            criteries.add(Criteria.where("client_id").is(clientId));
            return this;
        }

        public QueryBuilder withVideoId(String videoId) {
            criteries.add(Criteria.where("data.elements.options.video_id").is(videoId));
            return this;
        }

        public QueryBuilder withAudioId(String audioId) {
            criteries.add(Criteria.where("data.elements.options.audio_id").is(audioId));
            return this;
        }

        public QueryBuilder withPresetIds(List<Long> presetIds) {
            //we have preset_id of both types in mongodb (bloody hell!) - string and integer. So we have to search
            // through all types
            Object[] converted = new Object[presetIds.size() * 2];

            for (int i = 0; i < presetIds.size(); i++) {
                converted[2 * i] = presetIds.get(i);
                converted[2 * i + 1] = presetIds.get(i).toString();
            }

            criteries.add(Criteria.where("preset_id").in(converted));
            return this;
        }

        public QueryBuilder withArchive(Boolean status) {
            criteries.add(Criteria.where("archive").is(status));
            return this;
        }

        public QueryBuilder withStatusRtb(RtbStatus status) {
            criteries.add(Criteria.where("status_rtb").is(status));
            return this;
        }

        public QueryBuilder withNameRegexp(String regexp, String opts) {
            criteries.add(Criteria.where("name").regex(regexp, opts));
            return this;
        }

        public QueryBuilder withDateBetween(Date from, Date to) {
            criteries.add(Criteria.where("date").lt(to).gt(from));
            return this;
        }

        @Override
        public Query build() {
            Query query = new Query();

            for (Criteria criteria : criteries) {
                query.addCriteria(criteria);
            }

            return query;
        }
    }

    public static class UpdateBuilder {
        Update update = new Update();


        public UpdateBuilder withVast(String vast) {
            update.set("vast", vast);
            return this;
        }

        public UpdateBuilder withName(String name) {
            update.set("name", name);
            return this;
        }

        public UpdateBuilder withScreenshotUrl(String screenshotUrl) {
            update.set("screenshot_url", screenshotUrl);
            return this;
        }

        public UpdateBuilder withScreenshotIsDone(Boolean screenshotIsDone) {
            update.set("screenshot_is_done", screenshotIsDone);
            return this;
        }

        private Update build() {
            return update;
        }
    }

    public Addition findById(String id) {
        Query query = new QueryBuilder().withId(id).build();
        return databaseWrapper(() -> mongoOperations.findOne(query, Addition.class), "");
    }

    public Addition findByIdAndQuery(String id, QueryBuilder queryBuilder) {
        Query query = queryBuilder.build();
        query.addCriteria(Criteria.where("_id").is(id));
        return databaseWrapper(() -> mongoOperations.findOne(query, Addition.class), "");
    }

    public List<Addition> findByQuery(QueryBuilder queryBuilder) {
        return databaseWrapper(() -> mongoOperations.find(queryBuilder.build(), Addition.class), "find");
    }

    public ItemsWithTotal<Addition> getVideoAdditionsWithTotalByQuery(QueryBuilder queryBuilder,
                                                                      Integer limit,
                                                                      Integer offset,
                                                                      Sort.Direction sortOrder,
                                                                      Function<List<Addition>, Set<String>> prepare,
                                                                      BiFunction<Addition, Set<String>, Boolean> filter,
                                                                      Predicate<List<Addition>> isDone
    ) {
        Query query = queryBuilder.build();

        long total = databaseWrapper(() -> mongoOperations.count(query, Addition.class), "count");

        query.skip(offset).with(Sort.by(sortOrder, "date"));

        List<Addition> additions;

        long realOffset = offset;

        if (filter == null) {
            query.limit(limit);
            additions = databaseWrapper(() -> mongoOperations.find(query, Addition.class), "find");
            realOffset += additions.size();
        } else {
            additions = new ArrayList<>();

            if (isDone == null) {
                isDone = list -> list.size() >= limit;
            }

            Predicate<List<Addition>> finalIsDone = isDone;

            realOffset += databaseWrapper(
                    () -> filterStream(additions, finalIsDone, prepare, query, filter),
                    "stream"
            );
        }

        return new ItemsWithTotal<>(additions, total, realOffset);
    }

    private long filterStream(List<Addition> additions, Predicate<List<Addition>> isDone,
                              Function<List<Addition>, Set<String>> prepare, Query query,
                              BiFunction<Addition, Set<String>, Boolean> filter) {
        long realOffset = 0;

        try (CloseableIterator<Addition> cursor = mongoOperations.stream(query, Addition.class)) {
            List<Addition> buffer = new ArrayList<>();

            while (cursor.hasNext() && !isDone.test(additions)) {

                while (buffer.size() < 50 && cursor.hasNext()) {
                    buffer.add(cursor.next());
                }

                Set<String> prepared = null;

                if (prepare != null) {
                    prepared = prepare.apply(buffer);
                }

                for (Addition addition : buffer) {
                    if (isDone.test(additions)) {
                        break;
                    }

                    realOffset++;

                    if (filter.apply(addition, prepared)) {
                        additions.add(addition);
                    }
                }

                buffer.clear();
            }
        }

        return realOffset;
    }

    public UpdateResult update(QueryBuilder queryBuilder, UpdateBuilder updateBuilder) {
        return databaseWrapper(() -> mongoOperations.updateFirst(queryBuilder.build(), updateBuilder.build(), Addition.class), "update");
    }

    public UpdateResult updateVastById(String id, String vast) {
        return databaseWrapper(() -> mongoOperations
                .updateFirst(new Query(Criteria.where("_id").is(id)), Update.update("vast", vast), Addition.class), "update");
    }

    public UpdateResult updateVastByCreativeId(Long creativeId, String vast) {
        return databaseWrapper(() -> mongoOperations
                .updateFirst(new Query(Criteria.where("stock_creative_id").is(creativeId)), Update.update("vast", vast),
                        Addition.class), "update");
    }

    public UpdateResult updateScreenshotUrl(String id, String url) {
        return databaseWrapper(() -> mongoOperations
                .updateFirst(new Query(Criteria.where("_id").is(id)),
                        Update.update("screenshot_url", url), Addition.class), "update");
    }

    public Addition getAdditionById(String id) {
        return findByIdAndQuery(id, new QueryBuilder().withArchive(false));
    }

    public Addition getAdditionByIdArchivedAlso(String id) {
        return findById(id);
    }

    public Addition createAddition(Addition addition) {
        createAdditions(Collections.singletonList(addition));
        return addition;
    }

    public List<Addition> createAdditions(List<Addition> additions) {
        return databaseWrapper(() -> RepositoryUtils.insertWithDups(additions, mongoOperations, Addition.class), "insert");
    }

    public UpdateResult updateScreenshotUrl(final long creativeId, final String screenshotUrl,
                                            final Boolean screenshotIsDone) {
        return update(
                new VideoAdditionsRepository.QueryBuilder()
                        .withCreativeId(creativeId),
                new VideoAdditionsRepository.UpdateBuilder()
                        .withScreenshotUrl(screenshotUrl)
                        .withScreenshotIsDone(screenshotIsDone));
    }

    protected <REC> REC databaseWrapper(Supplier<REC> wrapper, String opName) {
        try (TraceProfile ignored = Trace.current().profile("db:" + opName, "canvas_video_additions")) {
            return wrapper.get();
        }
    }

    public UpdateResult updateStatusRtb(List<Long> creativeIds, RtbStatus statusRtb) {
        return databaseWrapper(() -> mongoOperations
                .updateMulti(new VideoAdditionsRepository.QueryBuilder()
                                .withCreativeIds(creativeIds)
                                .build(),
                        Update.update("status_rtb", statusRtb), Addition.class), "update");
    }
}
