package ru.yandex.canvas.repository.video;

import java.util.List;
import java.util.function.Supplier;

import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import ru.yandex.canvas.model.video.vc.feed.VideoConstructorFeed;
import ru.yandex.canvas.model.video.vc.feed.VideoConstructorFeeds;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class VideoConstructorFeedsRepository {
    private MongoOperations mongoOperations;

    public VideoConstructorFeedsRepository(MongoOperations mongoOperations) {
        this.mongoOperations = mongoOperations;
    }

    protected <REC> REC databaseWrapper(Supplier<REC> wrapper, String opName) {
        try (TraceProfile ignore = Trace.current().profile("db:" + opName, "canvas_video_constructor_feeds")) {
            return wrapper.get();
        }
    }

    public boolean delete(String feedId, Long clientId) {
        return databaseWrapper(() -> mongoOperations.updateFirst(
                new QueryBuilder().withId(feedId).withClientId(clientId).build(),
                Update.update("archive", true),
                VideoConstructorFeed.class).getModifiedCount(), "delete") > 0;
    }

    public VideoConstructorFeed save(VideoConstructorFeed record) {
        return databaseWrapper(() -> mongoOperations.save(record), "save");
    }

    public VideoConstructorFeed findById(String id, Long clientId) {
        return databaseWrapper(() -> mongoOperations.findOne(
                new QueryBuilder().withId(id).withClientId(clientId).build(),
                VideoConstructorFeed.class), "find");
    }

    public List<VideoConstructorFeed> findByIds(List<String> ids, Long clientId) {
        return databaseWrapper(() -> mongoOperations.find(
                new QueryBuilder().withIds(ids).withClientId(clientId).build(),
                VideoConstructorFeed.class), "find");
    }

    public VideoConstructorFeed findByIdInternal(String id) {
        return databaseWrapper(() -> mongoOperations.findOne(
                new QueryBuilder().withId(id).build(),
                VideoConstructorFeed.class), "find");
    }

    public VideoConstructorFeeds find(final long clientId, final boolean descOrder) {
        QueryBuilder queryBuilder = new QueryBuilder().withClientId(clientId).withArchive(false);
        Query query = queryBuilder.build();

        final long total = databaseWrapper(() -> mongoOperations.count(query, VideoConstructorFeed.class), "count");

        List<VideoConstructorFeed> files = databaseWrapper(() -> mongoOperations.find(query
                        .with(Sort.by(descOrder ? Sort.Direction.DESC : Sort.Direction.ASC, "creation_time")),
                VideoConstructorFeed.class), "find");

        return new VideoConstructorFeeds(files, total);
    }

    public static class QueryBuilder extends QueryBuilderBase<QueryBuilder> {
        public VideoConstructorFeedsRepository.QueryBuilder withId(String id) {
            criteries.add(Criteria.where("_id").is(id));
            return this;
        }

        public VideoConstructorFeedsRepository.QueryBuilder withIds(List<String> ids) {
            criteries.add(Criteria.where("_id").in(ids));
            return this;
        }

        public VideoConstructorFeedsRepository.QueryBuilder withClientId(Long clientId) {
            criteries.add(Criteria.where("client_id").is(clientId));
            return this;
        }

        public VideoConstructorFeedsRepository.QueryBuilder withArchive(Boolean status) {
            criteries.add(Criteria.where("archive").is(status));
            return this;
        }

        @Override
        public Query build() {
            Query query = new Query();
            for (Criteria criteria : criteries) {
                query.addCriteria(criteria);
            }
            return query;
        }
    }
}
