package ru.yandex.canvas.repository.video;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.mongodb.client.result.UpdateResult;
import org.apache.commons.lang3.StringUtils;
import org.bson.Document;
import org.bson.types.ObjectId;
import org.jetbrains.annotations.NotNull;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.vc.files.VideoConstructorFile;
import ru.yandex.canvas.model.video.vc.files.VideoConstructorFiles;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.SandBoxService;
import ru.yandex.canvas.service.video.Ratio;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class VideoConstructorFilesRepository {
    private MongoOperations mongoOperations;
    private DateTimeService dateTimeService;

    public VideoConstructorFilesRepository(MongoOperations mongoOperations, DateTimeService dateTimeService) {
        this.mongoOperations = mongoOperations;
        this.dateTimeService = dateTimeService;
    }

    protected <REC> REC databaseWrapper(Supplier<REC> wrapper, String opName) {
        try (TraceProfile ignore = Trace.current().profile("db:" + opName, "canvas_video_files")) {
            return wrapper.get();
        }
    }

    public boolean deleteFile(String fileId, Long clientId) {
        return databaseWrapper(() -> mongoOperations.updateFirst(
                new VideoConstructorFilesRepository.QueryBuilder().withId(fileId).withClientId(clientId).build(),
                Update.update("archive", true),
                VideoConstructorFile.class)
                .getModifiedCount(), "update") > 0;
    }

    public VideoConstructorFile upsert(VideoConstructorFile record) {
        if (record.getId() == null) {
            record.withId(new ObjectId().toString());
        }
        return save(record);
    }

    public VideoConstructorFile save(VideoConstructorFile record) {
        Query query = new VideoConstructorFilesRepository.QueryBuilder()
                .withId(record.getId())
                .build();

        Update update = new Update();
        Document document = new Document();
        mongoOperations.getConverter().write(record, document);
        for (Map.Entry<String, Object> field : document.entrySet()) {
            update.set(field.getKey(), field.getValue());
        }

        return databaseWrapper(() -> upsertAndGet(query, update, record.getClientId()), "save");
    }

    private VideoConstructorFile upsertAndGet(Query query, Update update, Long clientId) {
        UpdateResult result = mongoOperations.upsert(query, update, VideoConstructorFile.class);

        if (result.getModifiedCount() > 0) {
            return mongoOperations.findOne(query, VideoConstructorFile.class);
        } else {
            String id = result.getUpsertedId().asObjectId().getValue().toString();
            return findById(id, clientId);
        }
    }

    public VideoConstructorFile findById(String id, Long clientId) {
        return databaseWrapper(() -> mongoOperations.findOne(
                new VideoConstructorFilesRepository.QueryBuilder().withId(id).withClientId(clientId).build(),
                VideoConstructorFile.class), "find");
    }

    public List<VideoConstructorFile> findByIdsInternal(List<String> ids) {
        return databaseWrapper(() -> mongoOperations.find(
                new VideoConstructorFilesRepository.QueryBuilder().withIds(ids).build(),
                VideoConstructorFile.class), "find");
    }

    public VideoConstructorFile findByIdInternal(String id) {
        return databaseWrapper(() -> mongoOperations.findOne(
                new VideoConstructorFilesRepository.QueryBuilder().withId(id).build(),
                VideoConstructorFile.class), "find");
    }

    public void markFileUsed(String id, Long clientId) {
        Update update = Update.update("archive", false).set("date", dateTimeService.getCurrentDate());
        Query q = new VideoConstructorFilesRepository.QueryBuilder().withClientId(clientId).withId(id).build();
        databaseWrapper(() -> mongoOperations.upsert(q, update, VideoConstructorFile.class), "upsert");
    }

    public boolean updateFileFromSandboxTaskResult(String id,
                                                   SandBoxService.SandboxVideoConstructorTaskOutput taskResult,
                                                   @Nullable FileStatus fileStatus,
                                                   @Nullable String feedResultUrl) {
        Update update = Update.update("date", dateTimeService.getCurrentDate());
        if (taskResult.getMp4Url() != null) {
            update.set("mp4Url", taskResult.getMp4Url());
        }
        if (taskResult.getMp4StillageId() != null) {
            update.set("mp4_stillage_id", taskResult.getMp4StillageId());
        }
        if (taskResult.getPackshotUrl() != null) {
            update.set("packshotUrl", taskResult.getPackshotUrl());
        }
        if (taskResult.getPackshotStillageId() != null) {
            update.set("packshot_stillage_id", taskResult.getPackshotStillageId());
        }
        if (taskResult.getPreviewUrl() != null) {
            update.set("previewUrl", taskResult.getPreviewUrl());
        }
        if (taskResult.getPackshotStillageId() != null) {
            update.set("preview_stillage_id", taskResult.getPackshotStillageId());
        }
        if (taskResult.getFeedArchiveUrl() != null) {
            update.set("feed_archive_url", taskResult.getFeedArchiveUrl());
        }
        if (taskResult.getFeedArchiveUrls() != null) {
            update.set("feed_archive_urls", taskResult.getFeedArchiveUrls());
        }
        if (taskResult.getFeedMp4Urls() != null) {
            update.set("feed_mp4_urls", taskResult.getFeedMp4Urls());
        }
        if (fileStatus != null) {
            update.set("status", fileStatus);
        }
        if (feedResultUrl != null) {
            update.set("feed_result_url", feedResultUrl);
        }
        Query q = new VideoConstructorFilesRepository.QueryBuilder().withId(id).build();
        return databaseWrapper(() -> mongoOperations.updateFirst(q, update, VideoConstructorFile.class),
                "updateFile").getModifiedCount() == 1;
    }

    public long count(VideoConstructorFilesRepository.QueryBuilder queryBuilder) {
        return databaseWrapper(() -> mongoOperations.count(queryBuilder.build(), VideoConstructorFile.class), "count");
    }

    public List<VideoConstructorFile> findByQuery(VideoConstructorFilesRepository.QueryBuilder queryBuilder,
                                                  Sort.Direction sortDirection,
                                                  Integer limit,
                                                  Integer offset) {
        Query query = queryBuilder.build();
        if (limit != null) {
            query.limit(limit);
        }
        if (offset != null) {
            query.skip(offset);
        }
        if (sortDirection != null) {
            Sort sort = Sort.by(sortDirection, "date");
            query.with(sort);
        }

        return databaseWrapper(() -> mongoOperations.find(query, VideoConstructorFile.class), "find");
    }

    private static String buildConditionByRatioWithDurations(List<String> ratios, Map<String,
            List<Double>> ratioDurations, Double ratioDirationDelta) {
        // строим условие на javascript, так как монга не умеет в математические операции
        // операции простые, поэтому должно работать быстрее, чем фильтрация на бекенде
        String mongoDurationParamName = "this.duration";
        return ratios.stream()
                .map(ratio -> {
                    String conditionDurationJs = "";
                    if (ratioDurations != null && ratioDurations.containsKey(ratio)) {
                        conditionDurationJs = ratioDurations.get(ratio).stream()
                                .map(duration -> {
                                    if (ratioDirationDelta != null) {
                                        return String.format("(%s >= %.2f && %s <= %.2f)", mongoDurationParamName,
                                                duration - ratioDirationDelta, mongoDurationParamName,
                                                duration + ratioDirationDelta);
                                    } else {
                                        return String.format("%s == %.2f", mongoDurationParamName, duration);
                                    }
                                })
                                .collect(Collectors.joining(" || "));
                        conditionDurationJs = "&& (" + conditionDurationJs + ")";
                    }
                    String[] ratioParts = ratio.split(":", 2);
                    return String.format("(%s*this.width/this.height/%s == 1 %s)", ratioParts[1], ratioParts[0],
                            conditionDurationJs);
                })
                .collect(Collectors.joining(" || "));
    }

    private static String buildConditionByRatiosBetween(Ratio ratioFrom, Ratio ratioTo) {
        return String.format("(this.width * %d >= %d * this.height && this.width * %d <= %d * this.height)",
                ratioTo.getHeight(), ratioTo.getWidth(), ratioFrom.getHeight(), ratioFrom.getWidth());
    }

    private static Criteria buildWhereCondition(String conditionJs) {
        return new Criteria() {
            @NotNull
            @Override
            public Document getCriteriaObject() {
                return new Document("$where", "return " + conditionJs);
            }
        };
    }

    public VideoConstructorFiles find(final long clientId, final String name, final boolean descOrder,
                                      final int offset, final int limit, List<String> ratios,
                                      Ratio ratioFrom, Ratio ratioTo,
                                      Double durationFrom, Double durationTo, Boolean hasAudio, FileStatus fileStatus,
                                      Map<String, List<Double>> ratioDurations, Double ratioDirationDelta,
                                      Boolean hasFeed) {
        Criteria criteria;

        if (StringUtils.isNotBlank(name)) {
            criteria = Criteria.where("client_id").is(clientId).and("archive").is(false)
                            .and("name").regex(".*" + name + ".*");
        } else {
            criteria = Criteria.where("client_id").is(clientId)
                    .and("archive").is(false);
        }

        String conditionJs = null;
        if (ratioFrom != null && ratioTo != null) {
            conditionJs = buildConditionByRatiosBetween(ratioFrom, ratioTo);
        } else if (ratios != null && !ratios.isEmpty()) {
            conditionJs = buildConditionByRatioWithDurations(ratios, ratioDurations, ratioDirationDelta);
        }
        if (conditionJs != null && !conditionJs.isEmpty()) {
            Criteria whereCriteria = buildWhereCondition(conditionJs);
            criteria = criteria.andOperator(whereCriteria);
        }

        if (durationFrom != null || durationTo != null) {
            criteria = criteria.and("duration");
            if (durationFrom != null) {
                criteria = criteria.gte(durationFrom);
            }
            if (durationTo != null) {
                criteria = criteria.lte(durationTo);
            }
        }
        if (hasAudio != null) {
            criteria = criteria.and("audio_url").exists(hasAudio);
        }
        if (fileStatus != null) {
            criteria = criteria.and("status").is(fileStatus.getValue());
        }
        if (hasFeed != null) {
            criteria = criteria.and("feed_id").exists(hasFeed);
        }
        final Query query = new Query(criteria);

        final long total = mongoOperations.count(query, VideoConstructorFile.class);

        List<VideoConstructorFile> files = mongoOperations.find(query
                        .with(Sort.by(descOrder ? Sort.Direction.DESC : Sort.Direction.ASC, "creation_time"))
                        .skip(offset).limit(limit),
                VideoConstructorFile.class);

        return new VideoConstructorFiles(files, total);
    }

    public static class QueryBuilder extends QueryBuilderBase<VideoConstructorFilesRepository.QueryBuilder> {

        public VideoConstructorFilesRepository.QueryBuilder withId(String id) {
            criteries.add(Criteria.where("_id").is(id));
            return this;
        }

        public VideoConstructorFilesRepository.QueryBuilder withIds(List<String> ids) {
            criteries.add(Criteria.where("_id").in(ids));
            return this;
        }

        public VideoConstructorFilesRepository.QueryBuilder withClientId(Long clientId) {
            criteries.add(Criteria.where("client_id").is(clientId));
            return this;
        }

        public VideoConstructorFilesRepository.QueryBuilder withName(String name) {
            criteries.add(Criteria.where("name").is(name));
            return this;
        }

        public VideoConstructorFilesRepository.QueryBuilder withArchive(Boolean status) {
            criteries.add(Criteria.where("archive").is(status));
            return this;
        }

        public VideoConstructorFilesRepository.QueryBuilder withDateFrom(Date from) {
            criteries.add(Criteria.where("date").gt(from));
            return this;
        }

        public VideoConstructorFilesRepository.QueryBuilder withFeedExists(Boolean feedExists) {
            criteries.add(Criteria.where("feed_id").exists(feedExists));
            return this;
        }

        @Override
        public Query build() {
            Query query = new Query();
            for (Criteria criteria : criteries) {
                query.addCriteria(criteria);
            }
            return query;
        }
    }
}
