package ru.yandex.canvas.repository.video;

import java.math.BigInteger;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;

import com.google.common.collect.ImmutableSet;
import com.mongodb.client.result.UpdateResult;
import org.bson.Document;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.retry.annotation.Retryable;

import ru.yandex.canvas.model.video.VideoFiles;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.files.FileType;
import ru.yandex.canvas.model.video.files.MediaDataSource;
import ru.yandex.canvas.model.video.files.VideoSource;
import ru.yandex.canvas.service.DateTimeService;
import ru.yandex.canvas.service.video.Ratio;
import ru.yandex.canvas.service.video.VideoCreativeType;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

public class VideoFilesRepository {
    private MongoOperations mongoOperations;
    private DateTimeService dateTimeService;

    public VideoFilesRepository(MongoOperations mongoOperations, DateTimeService dateTimeService) {
        this.mongoOperations = mongoOperations;
        this.dateTimeService = dateTimeService;
    }

    public VideoFiles findByIdAndQuery(String id, QueryBuilder queryBuilder) {
        Query query = queryBuilder.build();
        query.addCriteria(Criteria.where("_id").is(id));

        return databaseWrapper(() -> mongoOperations.findOne(query, VideoFiles.class), "find");
    }

    public VideoFiles findById(String id) {
        QueryBuilder queryBuilder = new QueryBuilder();
        return findByIdAndQuery(id, queryBuilder);
    }

    public long count(QueryBuilder queryBuilder) {
        return databaseWrapper(() -> mongoOperations.count(queryBuilder.build(), VideoFiles.class), "count");
    }

    public List<VideoFiles> findByQuery(QueryBuilder queryBuilder, Sort.Direction sortDirection, int limit,
                                        int offset) {
        Query query = queryBuilder.build();
        query.limit(limit);
        query.skip(offset);

        if (sortDirection != null) {
            Sort sort = Sort.by(sortDirection, "date");
            query.with(sort);
        }

        return databaseWrapper(() -> mongoOperations.find(query, VideoFiles.class), "find");
    }

    protected <REC> REC databaseWrapper(Supplier<REC> wrapper, String opName) {
        try (TraceProfile profile = Trace.current().profile("db:" + opName, "canvas_video_files")) {
            return wrapper.get();
        }
    }

    public List<VideoFiles> findByQuery(QueryBuilder queryBuilder) {
        Query query = queryBuilder.build();
        return databaseWrapper(() -> mongoOperations.find(query, VideoFiles.class), "find");
    }

    public static class QueryBuilder extends QueryBuilderBase<QueryBuilder> {

        public QueryBuilder withId(String id) {
            criteries.add(Criteria.where("_id").is(id));
            return this;
        }

        public QueryBuilder withClientId(Long clientId) {
            criteries.add(Criteria.where("client_id").is(clientId));
            return this;
        }

        public QueryBuilder withStockFileId(String stockFileId) {
            criteries.add(Criteria.where("stock_file_id").is(stockFileId));
            return this;
        }

        public QueryBuilder withStockFile() {
            criteries.add(Criteria.where("stock_file_id").ne(null));
            return this;
        }

        public QueryBuilder withType(FileType type) {
            criteries.add(Criteria.where("type").is(type));
            return this;
        }

        public QueryBuilder withName(String name) {
            criteries.add(Criteria.where("name").is(name));
            return this;
        }

        public QueryBuilder withStatus(FileStatus... statuses) {
            criteries.add(Criteria.where("status").in((Object[]) statuses));
            return this;
        }

        public QueryBuilder withStatusNotIn(FileStatus... statuses) {
            criteries.add(Criteria.where("status").nin((Object[]) statuses));
            return this;
        }

        public QueryBuilder withStatusNe(FileStatus status) {
            criteries.add(Criteria.where("status").ne(status));
            return this;
        }

        public QueryBuilder withArchive(Boolean status) {
            criteries.add(Criteria.where("archive").is(status));
            return this;
        }

        public QueryBuilder withDuration(Long value) {
            criteries.add(Criteria.where("duration").is(value));
            return this;
        }

        public QueryBuilder withDurationBetween(double min, double max) {
            criteries.add(Criteria.where("duration").gte(min).lte(max));
            return this;
        }

        public QueryBuilder withNameRegexp(String regexp, String opts) {
            criteries.add(Criteria.where("name").regex(regexp, opts));
            return this;
        }

        public QueryBuilder withNullRatio() {
            criteries.add(Criteria.where("ratio_percent").is(null));
            return this;
        }

        public QueryBuilder withRatiosBetween(Ratio ratioFrom, Ratio ratioTo) {
            criteries.add(Criteria.where("ratio_percent").gte(ratioTo.ratioPercent()).lte(ratioFrom.ratioPercent()));
            return this;
        }

        public QueryBuilder withRatios(List<String> rations) {
            if (!rations.isEmpty()) {
                criteries.add(Criteria.where("ratio").in(rations));
            }
            return this;
        }

        public QueryBuilder withHeightGreaterThan(Long minHeight) {
            criteries.add(Criteria.where("stillage_info.metadataInfo.videoStreams.height").gte(minHeight));
            return this;
        }

        public QueryBuilder withNonNullConversionTaskId() {
            criteries.add(Criteria.where("conversion_task_id").ne(null));
            return this;
        }

        public QueryBuilder withConversionTaskId(Long convertionTaskId) {
            criteries.add(Criteria.where("conversion_task_id").is(convertionTaskId));
            return this;
        }

        public QueryBuilder withNonNullVideoMetaId() {
            criteries.add(Criteria.where("video_meta_id").ne(null));
            return this;
        }

        public QueryBuilder withCreativeType(VideoCreativeType creativeType) {
            criteries.add(Criteria.where("creativeType").is(creativeType));
            return this;
        }

        public QueryBuilder withCreativeTypeNe(VideoCreativeType creativeType) {
            criteries.add(Criteria.where("creativeType").ne(creativeType));
            return this;
        }

        public QueryBuilder withCreativeTypes(VideoCreativeType... types) {
            criteries.add(Criteria.where("creativeType").in((Object[]) types));
            return this;
        }

        private String buildConditionByRatiosBetween(Ratio ratioFrom, Ratio ratioTo) {
            // this.formats[0]["width"] * 9 >= 16 *this.formats[0]["height"]
            return String.format(
                    "(this.formats[0][\"width\"] * %d >= %d * this.formats[0][\"height\"] && this.formats[0][\"width\"] * %d <= %d * this.formats[0][\"height\"])",
                    ratioTo.getHeight(), ratioTo.getWidth(), ratioFrom.getHeight(), ratioFrom.getWidth());
        }

        private Criteria buildWhereCondition(String conditionJs) {
            return new Criteria() {
                @NotNull
                @Override
                public Document getCriteriaObject() {
                    return new Document("$where", "return " + conditionJs);
                }
            };
        }

        public QueryBuilder withRatiosBetweenExact(Ratio ratioFrom, Ratio ratioTo) {
            var conditionJs = buildConditionByRatiosBetween(ratioFrom, ratioTo);

            criteries.add(buildWhereCondition(conditionJs));
            return this;
        }

        @Override
        public Query build() {
            Query query = new Query();

            for (Criteria criteria : criteries) {
                query.addCriteria(criteria);
            }

            return query;
        }

        public QueryBuilder withDateBefore(Date to) {
            criteries.add(Criteria.where("date").lt(to));
            return this;
        }

        public QueryBuilder withDateBetween(Date from, Date to) {
            criteries.add(Criteria.where("date").lt(to).gt(from));
            return this;
        }
    }

    public static class UpdateBuilder {
        Update update = new Update();


        public UpdateBuilder withStatus(FileStatus status) {
            update.set("status", status);
            return this;
        }

        public UpdateBuilder withConversionId(Long id) {
            update.set("conversion_task_id", id);
            return this;
        }

        public UpdateBuilder withRatioPercent(Integer ratioPercent) {
            update.set("ratio_percent", ratioPercent);
            return this;
        }

        public UpdateBuilder withRatio(String ratio) {
            update.set("ratio", ratio);
            return this;
        }


        public UpdateBuilder withVideoMetaId(BigInteger id) {
            update.set("video_meta_id", id);
            return this;
        }

        public UpdateBuilder withPlayerId(String id) {
            update.set("player_id", id);
            return this;
        }

        public UpdateBuilder withStrmPrefix(String prefix) {
            update.set("strm_prefix", prefix);
            return this;
        }

        public UpdateBuilder withFormats(List<VideoFiles.VideoFormat> formats) {
            update.set("formats", formats);
            return this;
        }

        public UpdateBuilder withName(String name) {
            update.set("name", name);
            return this;
        }

        public UpdateBuilder withThumbnail(VideoFiles.VideoThumbnail thumbnail) {
            update.set("thumbnail", thumbnail);
            return this;
        }

        public UpdateBuilder withFirstFrameUrl(String firstFrameUrl) {
            update.set("firstFrameUrl", firstFrameUrl);
            return this;
        }

        public UpdateBuilder withSignaturesUrl(String val) {
            update.set("signaturesUrl", val);
            return this;
        }
        public UpdateBuilder withInputWidth(Long inputWidth) {
            update.set("inputWidth", inputWidth);
            return this;
        }

        public UpdateBuilder withInputHeight(Long inputHeight) {
            update.set("inputHeight", inputHeight);
            return this;
        }

        public UpdateBuilder withThumbnailUrl(String url) {
            update.set("thumbnailUrl", url);
            return this;
        }

        private Update build() {
            return update;
        }

        public UpdateBuilder withArchive(boolean b) {
            update.set("archive", b);
            return this;
        }
    }

    public String markFileUsed(QueryBuilder queryBuilder) {
        Update update = Update.update("archive", false).set("date", dateTimeService.getCurrentDate());
        Query q = queryBuilder.build();

        UpdateResult updateResult;

        try {
            updateResult = doUpsertVideoFiles(q, update);
        } catch (DuplicateKeyException e) {
            return null; //It was already used
        }

        if (updateResult.getUpsertedId() != null) {
            return updateResult.getUpsertedId().asObjectId().getValue().toString();
        } else {
            return null;
        }
    }

    public String markFileUsed(MediaDataSource file, Long clientId) {
        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder();

        queryBuilder.withClientId(clientId);

        boolean isStock = file.getStockId() != null;

        if (isStock) {
            queryBuilder.withStockFileId(file.getStockId())
                    .withType(file.getSourceType())
                    .withName(file.getName());
        } else {
            queryBuilder.withId(file.getId());
        }

        String newId = markFileUsed(queryBuilder);

        if (newId == null) {
            if (!isStock) {
                newId = file.getId();
            } else {
                List<VideoFiles> files = findByQuery(queryBuilder, null, 1, 0);

                if (files.size() == 0) {
                    throw new IllegalStateException("File wasn't marked as used");
                }

                newId = files.get(0).getId();
            }
        } else {//для только что созданного "использованного стокового видео" важно сохранить соотношение сторон
            if (file instanceof VideoSource) {
                VideoSource videoSource = (VideoSource) file;
                var updateBuilder = new VideoFilesRepository.UpdateBuilder()
                        .withRatio(videoSource.getRatio())
                        .withRatioPercent(videoSource.getRatioPercent());
                update(newId, updateBuilder);
            }
        }

        return newId;
    }

    public boolean deleteFile(String fileId, Long clientId, FileType type) {
        VideoFilesRepository.UpdateBuilder updateBuilder = new VideoFilesRepository.UpdateBuilder()
                .withArchive(true);

        VideoFilesRepository.QueryBuilder queryBuilder = new VideoFilesRepository.QueryBuilder()
                .withId(fileId)
                .withType(type)
                .withClientId(clientId);

        return update(queryBuilder, updateBuilder) > 0;
    }

    @Retryable(DuplicateKeyException.class)
    private UpdateResult doUpsertVideoFiles(Query q, Update update) {
        return databaseWrapper(() -> mongoOperations.upsert(q, update, VideoFiles.class), "upsert");
    }

    public long update(String id, UpdateBuilder updateBuilder) {
        return databaseWrapper(() -> mongoOperations
                .updateFirst(Query.query(Criteria.where("_id").is(id)), updateBuilder.build(), VideoFiles.class)
                .getModifiedCount(), "update");
    }

    public long update(QueryBuilder queryBuilder, UpdateBuilder updateBuilder) {
        return databaseWrapper(() -> mongoOperations.updateFirst(queryBuilder.build(), updateBuilder.build(),
                VideoFiles.class)
                .getModifiedCount(), "update");
    }

    public VideoFiles save(VideoFiles file) {
        Query query = new Query()
                .addCriteria(Criteria.where("client_id").is(file.getClientId()))
                .addCriteria(Criteria.where("stillage_id").is(file.getStillageId()))
                .addCriteria(Criteria.where("stock_file_id").is(file.getStockFileId()))
                .addCriteria(Criteria.where("type").is(file.getType()))
                .addCriteria(Criteria.where("name").is(file.getName()));

        if (file.getDuration() != null && Math.abs(file.getDuration() - 15.0) < 0.01) {
            query.addCriteria(Criteria.where("duration").in(null, 15.0, file.getDuration()));
        } else {
            query.addCriteria(Criteria.where("duration").is(file.getDuration()));
        }

        Update update = new Update()
                .setOnInsert("status", file.getStatus())
                .setOnInsert("conversion_task_id", null)
                .setOnInsert("formats", file.getFormats())
                .setOnInsert("creation_time", dateTimeService.getCurrentDate());

        Set<String> fieldsToSetOnInsert = ImmutableSet.of("status", "conversion_task_id", "formats", "creation_time");

        Document document = new Document();
        mongoOperations.getConverter().write(file, document);

        for (Map.Entry<String, Object> field : document.entrySet()) {
            if (fieldsToSetOnInsert.contains(field.getKey())) {
                continue;
            }

            update.set(field.getKey(), field.getValue());
        }

        return databaseWrapper(() -> upsertAndGet(query, update), "save");
    }

    private VideoFiles upsertAndGet(Query query, Update update) {
        UpdateResult result = mongoOperations.upsert(query, update, VideoFiles.class);

        if (result.getModifiedCount() > 0) {
            return mongoOperations.findOne(query, VideoFiles.class);
        } else {
            String id = result.getUpsertedId().asObjectId().getValue().toString();
            return findByIdAndQuery(id, new QueryBuilder());
        }
    }

}
