package ru.yandex.canvas.service;

import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.google.common.base.Stopwatch;
import one.util.streamex.StreamEx;
import org.checkerframework.checker.nullness.qual.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;

import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.elements.Element;
import ru.yandex.canvas.model.presets.Preset;
import ru.yandex.canvas.model.presets.PresetItem;
import ru.yandex.canvas.service.preview.CreativePreviewBuilder;
import ru.yandex.canvas.service.screenshooters.CreativesScreenshooterHelperService;
import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.function.Function.identity;

public abstract class AbstractCreativeDocumentModifierImpl implements CreativeDocumentModifier {
    private static final Logger logger = LoggerFactory.getLogger(AbstractCreativeDocumentModifierImpl.class);
    private static final Marker PERFORMANCE = MarkerFactory.getMarker("PERFORMANCE");
    private final CreativePreviewBuilder creativePreviewBuilder;
    private final PreviewUrlBuilder previewUrlBuilder;
    private final Map<Integer, Preset> presetsById;
    private final Map<String, Integer> presetIdByBundleName;
    protected final ScreenshooterService screenshooterService;
    protected final StillageService stillageService;
    protected final CreativesScreenshooterHelperService screenshooterHelperService;

    public AbstractCreativeDocumentModifierImpl(CreativePreviewBuilder creativePreviewBuilder,
                                                PreviewUrlBuilder previewUrlBuilder,
                                                PresetsService presetsService,
                                                ScreenshooterService screenshooterService,
                                                StillageService stillageService,
                                                CreativesScreenshooterHelperService screenshooterHelperService) {
        this.creativePreviewBuilder = creativePreviewBuilder;
        this.previewUrlBuilder = previewUrlBuilder;
        this.screenshooterService = screenshooterService;
        this.stillageService = stillageService;
        this.screenshooterHelperService = screenshooterHelperService;

        List<Preset> presets = presetsService.getRawUntranslatedPresets();
        this.presetsById = StreamEx.of(presets).toMap(Preset::getId, identity());
        this.presetIdByBundleName = StreamEx.of(presets)
                .toMap(preset -> preset.getItems().get(0).getBundle().getName(), Preset::getId);
    }

    @Override
    public void processCreativeDocument(String batchName, CreativeDocument item, long clientId) {
        Stopwatch uploadFileStopWatch = Stopwatch.createStarted();
        item.setCreativeURL(uploadToStillage(batchName,
                creativePreviewBuilder.buildCreativeHtmlForPreview(item)));

        logger.info(PERFORMANCE, "create_batch:stillage:{}",
                uploadFileStopWatch.elapsed(MILLISECONDS));

        ScreenShooterScreenshot screenshot = screenshooterHelperService.getScreenshot(item, null, null);

        item.setScreenshotIsDone(screenshot.getIsDone());
        if (!item.getScreenshotIsDone()) {
            item.setScreenshotURL(previewUrlBuilder.buildScreenshotUrl(clientId, item));
        } else {
            item.setScreenshotURL(screenshot.getUrl());
        }

        PresetItem presetItem = getPresetItem(item);
        if (presetItem != null) {
            Set<String> availableElementTypes = StreamEx.of(presetItem.getElements())
                    .map(Element::getType)
                    .toSet();
            List<Element> filteredElements = StreamEx.of(item.getData().getElements())
                    .filter(element -> availableElementTypes.contains(element.getType()))
                    .toList();
            item.getData().setElements(filteredElements);
        }
    }

    protected String uploadToStillage(String name, String creativeHTML) {
        return stillageService.uploadFile(name, creativeHTML.getBytes(StandardCharsets.UTF_8)).getUrl();
    }

    /**
     * Получает описание креатива в исходном шаблоне по его ID.
     * Возвращает null если не может найти нужный шаблон или нужный item в этом шаблоне.
     */
    @Nullable
    private PresetItem getPresetItem(CreativeDocument creative) {
        if (creative.getData() == null || creative.getData().getBundle() == null
                || creative.getData().getBundle().getName() == null) {
            // дальше проверять нечего, ошибку на отсутствие этих полей должна выдать валиация по аннотациям этих полей
            return null;
        }

        Integer presetId = creative.getPresetId();

        // PresetId появился относительно недавно, поэтому большинство креативов его не имеют. Чтобы решить эту проблему
        // попытаемся его определить с помощью bundle name.
        if (presetId == null) {
            String bundleName = creative.getData().getBundle().getName();
            presetId = presetIdByBundleName.get(bundleName);
        }

        if (presetId == null) {
            return null;
        }

        Preset preset = presetsById.get(presetId);
        if (preset == null) {
            return null;
        }

        int itemId = (int) creative.getId();
        int anotherItemId = creative.getItemId();

        return preset.getItemById(anotherItemId) == null ? preset.getItemById(itemId) :
                preset.getItemById(anotherItemId);
    }

}
