package ru.yandex.canvas.service;

import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import com.google.common.base.Stopwatch;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;
import org.springframework.retry.annotation.Retryable;

import ru.yandex.canvas.exceptions.AuthException;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.UnauthorizedException;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.direct.asynchttp.ErrorResponseWrapperException;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.GET;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.Query;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

/**
 * Service that rule the permissions on backend. By default users has GUEST rigths and can see only previews
 * Additional rights comes from request-scoped bean AuthRequestParams.
 * WARN: Do not use auth features in standard executor's threads, cause it has no request thread local context.
 * TODO: Make custom thread pool executors with request scope
 *
 * @author solovyev
 */
public class AuthServiceHttpProxy implements AuthService {
    private static final Marker PERFORMANCE = MarkerFactory.getMarker("PERFORMANCE");
    private static final Logger logger = LoggerFactory.getLogger(AuthServiceHttpProxy.class);
    private final AuthRequestParams authRequestParams;
    private Api api;

    public AuthServiceHttpProxy(@NotNull AuthRequestParams authRequestParams,
                                @NotNull String authUrl, TvmIntegration tvmIntegration, TvmService tvmService,
                                ParallelFetcherFactory fetcherFactory) {
        this.authRequestParams = authRequestParams;
        this.api = Smart.builder()
                .withParallelFetcherFactory(fetcherFactory)
                .useTvm(tvmIntegration, tvmService)
                .withProfileName("auth_service_http_proxy")
                .withBaseUrl(authUrl)
                .build()
                .create(Api.class);
    }

    public interface Api {
        @GET("/ping")
        @Headers("Content-Type: application/json")
        Call<String> ping();

        @GET("/authenticate")
        @Json
        @Headers("Content-Type: application/json")
        Call<Privileges> authenticate(@Query("user_id") String userId, @Query("client_id") String clientId);
        //client_id строка специально. Иначе клиент не пошлёт null в запросе и пропустит обязательный параметр client_id
    }

    /**
     * WARN: Do not use auth features in standard executor's threads, cause it has no request thread local context.
     *
     * @return Privileges with a set of Permissions
     */
    @Override
    public Privileges authenticate() {
        final Long userId = authRequestParams.getUserId().orElse(null);
        final Long clientId = authRequestParams.getClientId().orElse(null);

        return authenticate(userId, clientId);
    }


    /**
     * WARN: Do not use auth features in standard executor's threads, cause it has no request thread local context.
     *
     * @return true if current user has specified permission on current client
     */
    @Override
    @Retryable(maxAttempts = 3)
    public boolean checkPermission(@NotNull Privileges.Permission permission) {
        return authenticate().checkPermissionOn(permission);
    }

    /**
     * WARN: Do not use auth features in standard executor's threads, cause it has no request thread local context.
     *
     * @return true if current user has specified permission on different client
     */
    @Override
    @Retryable(maxAttempts = 3)
    public boolean checkPermissionDifferentClient(long clientId, @NotNull Privileges.Permission permission) {
        final Long userId = authRequestParams.getUserId().orElse(null);

        return authenticate(userId, clientId).checkPermissionOn(permission);
    }

    /**
     * WARN: Do not use auth features in standard executor's threads, cause it has no request thread local context.
     *
     * @throws AuthException if current user has no specified permission on current client
     */
    @Override
    public void requirePermission(@NotNull Privileges.Permission permission) {
        if (!checkPermission(permission)) {
            throw new AuthException(permission.getRestrictedMsg());
        }
    }

    @Override
    public void requirePermissionDifferentClient(long clientId, @NotNull Privileges.Permission permission) {
        if (!checkPermissionDifferentClient(clientId, permission)) {
            throw new AuthException(permission.getRestrictedMsg());
        }
    }

    @Override
    public String ping() {
        logger.info("executing request to intapi \"{}\"", api.ping().getRequest());
        Result<String> result = api.ping().execute();
        if (result.getSuccess() == null && result.getErrors() != null) {
            logger.error("Depended ping failed: Auth service ({})", result.getErrors()
                    .stream()
                    .map(error -> error.toString() +
                            (error instanceof ErrorResponseWrapperException &&
                                    ((ErrorResponseWrapperException) error).getResponse() != null ?
                                    " (" + ((ErrorResponseWrapperException) error).getResponse().toString() + ")": ""))
                    .collect(Collectors.toList()));
            throw new InternalServerError("Depended ping failed");
        }
        return "OK";
    }

    @Override
    public long getUserId() {
        return authRequestParams.getUserId().orElseThrow(UnauthorizedException::new);
    }

    @NotNull
    private Privileges authenticate(Long userId, Long clientId) {
        final Stopwatch stopWatch = Stopwatch.createStarted();
        String clientIdParam = clientId == null ? "" : String.valueOf(clientId);
        String userIdParam = userId == null ? "" : String.valueOf(userId);
        try {
            logger.info("executing request to intapi \"{}\"", api.authenticate(userIdParam, clientIdParam).getRequest());
            Result<Privileges> result = api.authenticate(userIdParam, clientIdParam).execute();
            if (result.getSuccess() == null && result.getErrors() != null) {
                logger.error("Auth request was failed ({})", result.getErrors()
                        .stream()
                        .map(error -> error.toString() +
                                (error instanceof ErrorResponseWrapperException &&
                                        ((ErrorResponseWrapperException) error).getResponse() != null ?
                                        " (" + ((ErrorResponseWrapperException) error).getResponse().toString() + ")": ""))
                        .collect(Collectors.toList()));
                throw new InternalServerError("Auth request was failed");
            }
            return result.getSuccess();
        } finally {
            logger.info(PERFORMANCE, "auth_proxy:authenticate:{}", stopWatch.elapsed(TimeUnit.MILLISECONDS));
        }
    }
}
