package ru.yandex.canvas.service;

import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.retry.annotation.Retryable;

import ru.yandex.canvas.exceptions.AuthException;
import ru.yandex.canvas.exceptions.DirectAPIException;
import ru.yandex.canvas.model.direct.CreativeCampaignRequest;
import ru.yandex.canvas.model.direct.CreativeCampaignResult;
import ru.yandex.canvas.model.direct.CreativeUploadData;
import ru.yandex.canvas.model.direct.DirectUploadResult;
import ru.yandex.canvas.model.direct.DirectUploadableHelperService;
import ru.yandex.canvas.service.video.GetAccessToFeatureRequest;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Body;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.Query;
import ru.yandex.direct.tvm.TvmIntegration;
import ru.yandex.direct.tvm.TvmService;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.utils.JsonUtils.toJson;

/**
 * @author skirsanov
 */
public class DirectService {
    private static final Logger logger = LoggerFactory.getLogger(DirectService.class);

    private final String authToken;
    private final Api api;

    public DirectService(String serviceUrl,
                         String authToken,
                         TvmIntegration tvmIntegration,
                         TvmService tvmService,
                         AsyncHttpClient asyncHttpClient) {
        this.authToken = authToken;

        ParallelFetcherFactory fetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
        this.api = Smart.builder()
                .withParallelFetcherFactory(fetcherFactory)
                .withProfileName("direct_service")
                .withBaseUrl(serviceUrl)
                .useTvm(tvmIntegration, tvmService)
                .addHeaderConfigurator(headers -> headers.add(HttpHeaders.CONTENT_TYPE,
                        MediaType.APPLICATION_JSON_VALUE))
                .build()
                .create(Api.class);
    }

    public interface Api {
        @POST("/DisplayCanvas/upload_creatives")
        @Json
        @Headers("Content-Type: application/json")
        Call<DirectUploadResult> uploadCreatives(
                @Body @Json List<CreativeUploadData> uploadCreativesRequest,
                @Query("client_id") Long clientId,
                @Query("operator_uid") Long operatorUid);

        @POST("/DisplayCanvas/get_creatives_campaigns")
        @Json
        @Headers("Content-Type: application/json")
        Call<Map<Long, List<CreativeCampaignResult>>> getCreativesCampaigns(
                @Body @Json CreativeCampaignRequest getCreativeCampaignsRequest,
                @Query("client_id") Long clientId,
                @Query("operator_uid") Long operatorUid);

        @POST("/feature_dev/access")
        @Json
        @Headers("Content-Type: application/json")
        Call<FeatureResponse> getFeatures(
                @Body @Json GetAccessToFeatureRequest getFeaturesRequest);
    }

    @Retryable()
    public <T> DirectUploadResult sendCreatives(@Nullable final Long operatorUid, long clientId, List<T> creatives,
                                                DirectUploadableHelperService<T> service) {
        List<CreativeUploadData> directCreativeUploadData = creatives.stream()
                .map(creative -> service.toCreativeUploadData(creative, clientId))
                .collect(toList());

        Call<DirectUploadResult> uploadCall = api.uploadCreatives(directCreativeUploadData, clientId, operatorUid);

        String requestUrl = uploadCall.getRequest().getAHCRequest().getUrl();
        logger.info("direct POST request URL: {}, body: {} ", requestUrl, toJson(directCreativeUploadData));

        Result<DirectUploadResult> uploadResult = uploadCall.execute();
        checkResultForErrors(uploadResult, error -> new DirectAPIException("DIRECT_INTERNAL_ERROR: " + error));

        DirectUploadResult result = uploadResult.getSuccess();
        logger.info("direct response: {} ", result);

        if (!result.isOk()) {
            logger.error("direct requestUri: {}, req: {}, response: {} ", requestUrl, directCreativeUploadData, result);
            throw new DirectAPIException("DIRECT_VALIDATION_ERROR");
        }

        return result;
    }

    @Retryable()
    public Map<Long, List<CreativeCampaignResult>> getCreativesCampaigns(List<Long> creativesIds, Long uid,
                                                                         Long clientId) {
        CreativeCampaignRequest creativeCampaignRequest = new CreativeCampaignRequest().setCreativeIds(creativesIds);

        Call<Map<Long, List<CreativeCampaignResult>>> getCreativesCampaignsCall = api.getCreativesCampaigns(
                creativeCampaignRequest, clientId, uid);

        logger.info("campaigns requested {}", getCreativesCampaignsCall.getRequest().getAHCRequest().getUrl());

        Result<Map<Long, List<CreativeCampaignResult>>> getCreativesResult = getCreativesCampaignsCall.execute();
        checkResultForErrors(getCreativesResult, error -> new DirectAPIException("DIRECT_VALIDATION_ERROR: " + error));

        return getCreativesResult.getSuccess();
    }

    public void checkToken(final String token) {
        if (!this.authToken.equalsIgnoreCase(token)) {
            throw new AuthException("invalid token");
        }
    }

    @Retryable()
    public Set<String> getFeatures(Long clientId, Long uid) {
        if (clientId == null) {
            return Collections.emptySet();
        }

        GetAccessToFeatureRequest getAccessToFeatureRequest =
                new GetAccessToFeatureRequest().setClientIds(Collections.singletonList(clientId));

        Result<FeatureResponse> getFeaturesResult = api.getFeatures(getAccessToFeatureRequest).execute();
        checkResultForErrors(getFeaturesResult, error -> new DirectAPIException("DIRECT_VALIDATION_ERROR: " + error));

        List<String> features = getFeaturesResult.getSuccess().getResult().getClientIdsFeatures().get(clientId);

        if (features == null) {
            return Collections.emptySet();
        }

        return new HashSet<>(features);
    }
}
