package ru.yandex.canvas.service;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.function.IntFunction;
import java.util.function.Supplier;
import java.util.stream.IntStream;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.imageio.stream.ImageInputStream;
import javax.validation.constraints.NotNull;

import com.google.common.base.Stopwatch;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.mongodb.DuplicateKeyException;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.FindAndModifyOptions;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.retry.annotation.Retryable;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.multipart.MultipartFile;

import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.ConflictException;
import ru.yandex.canvas.exceptions.CropImageException;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.StillageFileNotFoundException;
import ru.yandex.canvas.model.CropParameters;
import ru.yandex.canvas.model.File;
import ru.yandex.canvas.model.Files;
import ru.yandex.canvas.model.avatars.AvatarsPutCanvasResult;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.stock.StockCategory;
import ru.yandex.canvas.model.stock.StockFile;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.service.TankerKeySet.ERROR;

/**
 * @author skirsanov, solovyev
 * FIXME: Change file upload permission to 'FILE' or 'CREATIVE_CREATE'
 */
public class FileService {
    private static final Logger logger = LoggerFactory.getLogger(FileService.class);
    private static final Marker PERFORMANCE = MarkerFactory.getMarker("PERFORMANCE");

    private final MongoOperations mongoOperation;
    private final StockService stockService;
    private final StillageService stillageService;
    private final AuthService authService;
    private final AvatarsService avatarsService;
    private final RestTemplate restTemplate;
    private final ExecutorService batchExecutor;

    public FileService(final MongoOperations mongoOperation,
                       final StockService stockService,
                       final StillageService stillageService,
                       final AuthService authService,
                       final AvatarsService avatarsService,
                       final RestTemplate restTemplate) {
        this.mongoOperation = mongoOperation;
        this.stockService = stockService;
        this.stillageService = stillageService;
        this.authService = authService;
        this.avatarsService = avatarsService;
        this.restTemplate = restTemplate;
        this.batchExecutor = Executors.newCachedThreadPool(new ThreadFactoryBuilder()
                .setNameFormat("file-service-%d").build());
    }

    private static Query findFileByIdQuery(@NotNull final String id, final long clientId) {
        return new Query(Criteria.where("id").is(id).andOperator(Criteria.where("clientId").is(clientId)
                .andOperator(Criteria.where("archive").is(false))));
    }

    public File uploadFile(final MultipartFile file, @Nullable final String originalFileId,
                           @Nullable final CropParameters cropParameters, final long clientId,
                           final boolean isTurbo, final boolean useNewCrop, boolean allowAnimatedGifs) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        return uploadFileInternal(file, originalFileId, cropParameters, clientId, isTurbo, useNewCrop, false,
                allowAnimatedGifs, true);
    }

    public Files uploadFiles(final List<MultipartFile> files, @Nullable final String originalFileId,
                             @Nullable final List<CropParameters> cropParameters, final long clientId,
                             final boolean useNewCrop, boolean allowAnimatedGifs, @Nullable ExecutorService executor) {
        List<File> uploadedFiles = uploadFilesInParallel(
                Math.max(files.size(), cropParameters == null ? 0 : cropParameters.size()),
                executor,
                fileNum -> uploadFileInternal(
                        files.get(fileNum),
                        originalFileId,
                        cropParameters == null ? null : cropParameters.get(fileNum),
                        clientId,
                        false,
                        useNewCrop,
                        false,
                        allowAnimatedGifs,
                        executor == null));
        return new Files(uploadedFiles, uploadedFiles.size());
    }

    public List<File> uploadFilesByTrustedUrls(final List<String> trustedUrls,
                                               final List<String> fileNames,
                                               @Nullable final List<String> originalFileIds,
                                               @Nullable final List<CropParameters> cropParameters,
                                               final long clientId,
                                               final boolean useNewCrop,
                                               final boolean isFromFeed,
                                               boolean allowAnimatedGifs,
                                               @Nullable ExecutorService executor) {
        return uploadFilesInParallel(
                Math.max(trustedUrls.size(), cropParameters == null ? 0 : cropParameters.size()),
                executor,
                fileNum -> uploadFileByTrustedUrlInternal(
                        trustedUrls.get(fileNum),
                        fileNames.get(fileNum),
                        originalFileIds == null ? null : originalFileIds.get(fileNum),
                        cropParameters == null ? null : cropParameters.get(fileNum),
                        clientId,
                        false,
                        useNewCrop,
                        isFromFeed,
                        allowAnimatedGifs,
                        executor == null));
    }

    /**
     * a parallel version for {@link #uploadFileInternal(String, MultipartFile, CropParameters, long)}
     *
     * @param cropParameters for each file. Must be same length as files. Ensured at {@link ru.yandex.canvas.controllers.FilesController#uploadFilesFromCropped(String, List, List, long, long)}
     * @return list of uploaded files
     */
    private List<File> uploadFilesInParallel(int filesCount,
                                             @Nullable ExecutorService executor,
                                             IntFunction<File> uploadFunction) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        // poor man's zip
        List<CompletableFuture<File>> futures =
                IntStream.range(0, filesCount)
                        .mapToObj(i -> CompletableFuture.supplyAsync(
                                () -> uploadFunction.apply(i),
                                executor != null ? executor : batchExecutor))
                        .collect(toList());

        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[]{}))
                .thenApply(v -> futures.stream().map(CompletableFuture::join) // IGNORE-BAD-JOIN DIRECT-149116
                        .collect(toList()))
                .join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    public File uploadFile(URL fileUrl, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        final String fileName = new java.io.File(fileUrl.getPath()).getName();
        final StillageFileInfo stillageFileInfo = stillageService.uploadFile(fileName, fileUrl);
        validateFile(stillageFileInfo);

        final AvatarsPutCanvasResult.SizesInfo sizes = avatarsService.upload(stillageFileInfo.getUrl()).getSizes();

        return saveFile(stillageFileInfo, sizes, null, fileName, clientId, null, null, null);
    }

    public File uploadFile(final String stillageFileId, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        final StillageFileInfo stillageFileInfo =
                stillageService.getById(stillageFileId).orElseThrow(StillageFileNotFoundException::new);
        validateFile(stillageFileInfo);

        final AvatarsPutCanvasResult.SizesInfo sizes = avatarsService.upload(stillageFileInfo.getUrl()).getSizes();

        return saveFile(stillageFileInfo, sizes, null, stillageFileId, clientId, null, null, null);
    }

    public File uploadIdeaFile(final String stillageFileId, String ideaId, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);
        authService.requirePermission(Privileges.Permission.IDEA);

        return uploadIdeaFileInternal(stillageFileId, ideaId, clientId);
    }

    /**
     * Internal function DO NOT check permissions and COULD BE called from any thread
     */
    public File uploadIdeaFileInternal(final String stillageFileId, String ideaId, final long clientId) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        final StillageFileInfo stillageFileInfo =
                stillageService.getById(stillageFileId).orElseThrow(StillageFileNotFoundException::new);
        validateFile(stillageFileInfo);
        logger.info(PERFORMANCE, "file_service:upload_idea_file_stillage:{}", stopwatch.elapsed(TimeUnit.MILLISECONDS));

        stopwatch.reset().start();
        final AvatarsPutCanvasResult.SizesInfo sizes = avatarsService.upload(stillageFileInfo.getUrl()).getSizes();

        File result = saveFile(stillageFileInfo, sizes, null, stillageFileId, clientId, null, ideaId, null);
        logger.info(PERFORMANCE, "file_service:upload_idea_file_avatar+mongo:{}",
                stopwatch.elapsed(TimeUnit.MILLISECONDS));
        return result;
    }

    public File saveFileFromStock(@NotNull final StockFile stockFile, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        Objects.requireNonNull(stockFile);

        final File file = new File();

        stockFile.getCategories().stream().findFirst()
                .flatMap(stockService::getCategory)
                .map(StockCategory::getName)
                .ifPresent(file::setName);

        file.setClientId(clientId);
        file.setDate(new Date());
        file.setStockFileId(stockFile.getId());

        file.setStillageFileInfo(stockFile.getStillageFileInfo());
        file.setStillageFileId(stockFile.getStillageFileInfo().getId());

        file.setUrl(stockFile.getUrl());
        file.setThumbnailUrl(stockFile.getThumbnailUrl());
        file.setPreviewUrl(stockFile.getPreviewUrl());
        file.setLargePreviewUrl(stockFile.getLargePreviewUrl());
        file.setCropSourceUrl(stockFile.getCropSourceUrl());
        return saveFile(file);
    }

    @SuppressWarnings("SameParameterValue")
    public File saveFile(final StillageFileInfo stillageFileInfo, final AvatarsPutCanvasResult.SizesInfo sizes,
                         @Nullable final String originalFileId, @Nullable final String fileName,
                         final long clientId, @Nullable CropParameters cropParameters, @Nullable String ideaId,
                         @Nullable Boolean isFromFeed) {
        final File file = new File();

        file.setClientId(clientId);
        file.setDate(new Date());
        file.setOriginalFileId(originalFileId);
        file.setCropParameters(cropParameters);
        file.setName(fileName);
        file.setIdeaId(ideaId);
        file.setIsFromFeed(isFromFeed);

        file.setStillageFileInfo(stillageFileInfo);
        file.setStillageFileId(stillageFileInfo.getId());

        file.setUrl(avatarsService.getReadServiceUri() + sizes.getOrig().getPath());
        file.setThumbnailUrl(avatarsService.getReadServiceUri() + sizes.getThumbnail().getPath());
        file.setPreviewUrl(avatarsService.getReadServiceUri() + sizes.getPreview().getPath());
        file.setLargePreviewUrl(avatarsService.getReadServiceUri() + sizes.getLargePreview().getPath());
        file.setCropSourceUrl(avatarsService.getReadServiceUri() + sizes.getCropSource().getPath());

        return saveFile(file);
    }

    /*
    Загрузить файл, который уже находится в памяти
     */
    private File uploadFileInternal(final MultipartFile file,
                                    @Nullable final String originalFileId,
                                    @Nullable final CropParameters cropParameters,
                                    final long clientId,
                                    final boolean isTurbo,
                                    final boolean useNewCrop,
                                    final boolean isFromFeed,
                                    boolean allowAnimatedGifs,
                                    boolean runInParallel) {
        if (file.isEmpty()) {
            throw new BadRequestException();
        }
        try {
            return uploadFileDataInternal(file.getBytes(), file.getOriginalFilename(), originalFileId, cropParameters,
                    clientId, isTurbo, useNewCrop, isFromFeed, allowAnimatedGifs, runInParallel);
        } catch (IOException e) {
            logger.error("error during upload", e);
            throw new InternalServerError();
        }
    }

    /*
    Загрузить файл по доверенной! ссылке, например из mds
     */
    private File uploadFileByTrustedUrlInternal(final String trustedUrl,
                                                final String fileName,
                                                @Nullable final String originalFileId,
                                                @Nullable final CropParameters cropParameters,
                                                final long clientId,
                                                final boolean isTurbo,
                                                final boolean useNewCrop,
                                                final boolean isFromFeed,
                                                boolean allowAnimatedGifs,
                                                boolean runInParallel) {
        byte[] fileData = restTemplate.getForObject(trustedUrl, byte[].class);
        return uploadFileDataInternal(fileData, fileName, originalFileId, cropParameters, clientId, isTurbo,
                useNewCrop, isFromFeed, allowAnimatedGifs, runInParallel);
    }

    /**
     * Internal function DO NOT check permissions and COULD BE called from any thread
     */
    private File uploadFileDataInternal(final byte[] fileData,
                                        final String fileName,
                                        @Nullable final String originalFileId,
                                        @Nullable final CropParameters cropParameters,
                                        final long clientId,
                                        final boolean isTurbo,
                                        final boolean useNewCrop,
                                        final boolean isFromFeed,
                                        final boolean allowAnimatedGifs,
                                        final boolean runInParallel) {
        final byte[] data;
        if (useNewCrop && cropParameters != null) {
            data = cropImage(fileData, cropParameters);
        } else {
            data = fileData;
        }

        final StillageFileInfo stillageFileInfo;
        final AvatarsPutCanvasResult.SizesInfo avatarsSizes;
        Supplier<StillageFileInfo> uploadToStillageSupplier = () -> stillageService.uploadFile(fileName, data);
        Supplier<AvatarsPutCanvasResult.SizesInfo> uploadToAvatarsSupplier =
                () -> avatarsService.upload(fileName, data).getSizes();

        // если хотим запустить параллельно, то запускаем операции в новых потоках
        if (runInParallel) {
            CompletableFuture<StillageFileInfo> uploadingToStillage = CompletableFuture.supplyAsync(
                    uploadToStillageSupplier, batchExecutor);
            CompletableFuture<AvatarsPutCanvasResult.SizesInfo> gettingSizes = CompletableFuture.supplyAsync(
                    uploadToAvatarsSupplier, batchExecutor);
            stillageFileInfo = uploadingToStillage.join(); // IGNORE-BAD-JOIN DIRECT-149116
            avatarsSizes = gettingSizes.join(); // IGNORE-BAD-JOIN DIRECT-149116
        } else {
            // если нет, то выполняем операции последовательно в текущем потоке
            stillageFileInfo = uploadToStillageSupplier.get();
            avatarsSizes = uploadToAvatarsSupplier.get();
        }

        //Валидация вынесена в тред servlet container'а, чтобы показывать сообщения об ошибках на корректном языке
        // (Thread local Locale)
        validateFile(stillageFileInfo, isTurbo, allowAnimatedGifs);
        return saveFile(stillageFileInfo, avatarsSizes, originalFileId, fileName, clientId, cropParameters,
                null, isFromFeed);
    }

    /*
    Делаем кроп исходного изображения и сохраняем в том же формате
     */
    private byte[] cropImage(byte[] origImageData, CropParameters cropParameters) {
        try {
            InputStream inputStream = new ByteArrayInputStream(origImageData);
            ImageInputStream imageInputStream = ImageIO.createImageInputStream(inputStream);
            Iterator<ImageReader> imageReaders = ImageIO.getImageReaders(imageInputStream);

            ImageReader reader = imageReaders.next();
            String format = reader.getFormatName();
            reader.setInput(imageInputStream, true, true);

            BufferedImage origImage;
            try {
                origImage = reader.read(0);
            } finally {
                reader.dispose();
                imageInputStream.close();
            }

            int x = (int) cropParameters.getX();
            int y = (int) cropParameters.getY();
            int width = (int) cropParameters.getWidth();
            int height = (int) cropParameters.getHeight();

            // если хотя бы одна точка прямоугольника для кропа лежит вне изображения,
            // то не кропаем и возвращаем исходное изображение (исключение не выкидываем)
            if (x < 0 || x >= origImage.getWidth() ||
                    y < 0 || y >= origImage.getHeight() ||
                    width <= 0 || x + width > origImage.getWidth() ||
                    height <= 0 || y + height > origImage.getHeight()) {
                return origImageData;
            }

            // Исторически сложилось, что параметры кропа приходят в вещественных числах, теперь же будут приходить
            // целые.
            // Создавать отдельное поле с целыми числами слишком дорого ради такого, поэтому просто берем целую часть.
            BufferedImage croppedImage = origImage.getSubimage(x, y, width, height);

            ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
            ImageIO.write(croppedImage, format, outputStream);
            return outputStream.toByteArray();
        } catch (Exception e) {
            throw new CropImageException("Can't crop image", e);
        }
    }

    /**
     * Upsert given {@link File} by it's {@link File#clientId} and {@link File#stillageFileId}.
     * <p>
     * For already existing files always un-archives and updates  {@link File#date} to current time.
     * If the file was not manually renamed via UI also updates it's name to newly given too.
     *
     * @param file the file data to save
     * @return upserted file
     * <p>
     * Retried on {@link DuplicateKeyException} and {@link org.springframework.dao.DuplicateKeyException}
     * due to a possible race condition with key collision.
     */
    @Retryable(value = {DuplicateKeyException.class, org.springframework.dao.DuplicateKeyException.class})
    private File saveFile(final File file) {
        final Stopwatch createFileStopWatch = Stopwatch.createStarted();

        final File existing = mongoOperation.findOne(
                // There is unique index on these fields
                new Query(Criteria
                        .where("clientId")
                        .is(file.getClientId().orElseThrow(() -> new IllegalStateException("clientId is required")))
                        .and("stillageFileId").is(file.getStillageFileId())
                        .and("name").is(file.getName())
                ),
                File.class);

        final File result;
        if (existing != null) {
            existing.setArchive(false);
            existing.setDate(new Date());
            if (file.getTurboParameters() != null) {
                existing.setTurboParameters(file.getTurboParameters());
            }
            result = existing;
        } else {
            result = file;
        }

        // There should be a proper explanation why this race condition does not pose any risk.
        // The point is that any kind of operation is idempotent and therefore the worst that can happen is
        // that the final state of a File may correspond not to the latest operation but to one of the previous
        // as if it was the latest actually. Therefore, no broken state may arise.
        mongoOperation.save(result);

        logger.info(PERFORMANCE, "create_file:mongo:{}", createFileStopWatch.elapsed(MILLISECONDS));

        return result;
    }

    public Optional<File> getById(final String id, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        final Stopwatch getFileMongoStopWatch = Stopwatch.createStarted();

        final Query query = new Query(Criteria.where("id").is(id).andOperator(Criteria.where("clientId").is(clientId)));

        final Optional<File> result = Optional.ofNullable(mongoOperation.findOne(query, File.class));

        logger.info(PERFORMANCE, "get_file:mongo:{}", getFileMongoStopWatch.elapsed(MILLISECONDS));

        return result;
    }

    public Optional<File> turbonizedById(final String id, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        final Stopwatch getFileMongoStopWatch = Stopwatch.createStarted();

        final Query query = new Query(Criteria.where("id").is(id).andOperator(Criteria.where("clientId").is(clientId)));

        File file = mongoOperation.findOne(query, File.class);

        if (file != null && file.getTurboParameters() == null) {
            turbonizeFile(file);
        }

        logger.info(PERFORMANCE, "get_file_turbo:mongo:{}", getFileMongoStopWatch.elapsed(MILLISECONDS));

        return Optional.ofNullable(file);
    }

    public void turbonizeFile(File file) {
        if (file.getTurboParameters() == null) {
            file.setTurboParameters(
                    avatarsService.uploadTurbo(file.getUrl()).toTurboParameters(avatarsService.getReadServiceUri()));
            saveFile(file);
        }
    }

    public Optional<File> getByIdInternal(final String id) {

        final Stopwatch getFileMongoStopWatch = Stopwatch.createStarted();

        final Optional<File> result = Optional.ofNullable(mongoOperation.findOne(new Query(Criteria.where("id").is(id)),
                File.class));

        logger.info(PERFORMANCE, "get_file_no_client:mongo:{}", getFileMongoStopWatch.elapsed(MILLISECONDS));

        return result;
    }

    public Files find(final long clientId, final String name, final boolean descOrder,
                      final int offset, final int limit, @Nullable final Boolean showFeeds) {
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        Criteria criteria = Criteria.where("clientId").is(clientId)
                .and("archive").is(false)
                .and("originalFileId").is(null);

        if (StringUtils.isNotBlank(name)) {
            criteria = criteria.and("name").regex(".*" + name + ".*");
        }
        if (showFeeds != null) {
            criteria = criteria.and("isFromFeed");
            if (showFeeds) {
                criteria = criteria.is(true);
            } else {
                criteria = criteria.ne(true);
            }
        }
        final Query query = new Query(criteria);

        final long total = mongoOperation.count(query, File.class);

        final Stopwatch findFileStopWatch = Stopwatch.createStarted();

        List<File> files = mongoOperation.find(query
                        .with(Sort.by(descOrder ? Sort.Direction.DESC : Sort.Direction.ASC, "date"))
                        .skip(offset).limit(limit),
                File.class);
        logger.info(PERFORMANCE, "find_files:mongo:{}", findFileStopWatch.elapsed(MILLISECONDS));

        return new Files(files, total);
    }

    /**
     * Only rename is supported
     *
     * @throws ConflictException if file with such name and stillageFileId is present for current client
     */
    public File update(@NotNull final File file, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        final Stopwatch saveFileStopWatch = Stopwatch.createStarted();

        final File result;
        try {
            result = mongoOperation.findAndModify(findFileByIdQuery(file.getId(), clientId),
                    new Update().set("name", file.getName()),
                    new FindAndModifyOptions().returnNew(true), File.class);
        } catch (org.springframework.dao.DuplicateKeyException e) {
            throw new ConflictException(ERROR.key("files-modify-conflict"), e);
        }
        logger.info(PERFORMANCE, "save_file:mongo:{}", saveFileStopWatch.elapsed(MILLISECONDS));

        return result;
    }

    void updateFilesDates(@NotNull final Collection<String> fileIDs, final long clientId) {
        if (!fileIDs.isEmpty()) {
            final Stopwatch updateFilesDatesStopWatch = Stopwatch.createStarted();

            mongoOperation.updateMulti(new Query(Criteria.where("id").in(fileIDs)
                            .and("clientId").is(clientId)),
                    new Update().set("date", new Date()).set("archive", false), File.class);

            logger.info(PERFORMANCE, "update_files_dates:mongo:{}", updateFilesDatesStopWatch.elapsed(MILLISECONDS));
        }
    }

    public void delete(@NotNull final String id, final long clientId) {
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);

        final Stopwatch deleteFileStopWatch = Stopwatch.createStarted();

        mongoOperation.updateFirst(findFileByIdQuery(id, clientId), new Update().set("archive", true), File.class);

        logger.info(PERFORMANCE, "delete_file:mongo:{}", deleteFileStopWatch.elapsed(MILLISECONDS));
    }

    private void validateFile(final StillageFileInfo stillageAnswer) {
        validateFile(stillageAnswer, false, false);
    }

    private void validateFile(final StillageFileInfo stillageAnswer, final boolean isTurbo, boolean allowAnimatedGifs) {
        logger.trace("validating uploaded file: {}", stillageAnswer);
        FileValidator validator = new ImageValidator(stillageAnswer, isTurbo, allowAnimatedGifs);
        validator.validate();
    }
}
