package ru.yandex.canvas.service;

import java.util.Set;

import com.google.common.collect.ImmutableSet;

import ru.yandex.canvas.model.stillage.StillageFileInfo;

import static org.springframework.util.MimeTypeUtils.IMAGE_GIF_VALUE;

//TODO This file should be a component and takes all his parameters as arguments of validate().
//It seems that there is no reasons to have a common interface (FileValidator) for all validators.
public class ImageValidator extends FileValidator {
    private static final int IMAGE_MAX_SIZE_MEGA_BYTES = 20;
    private static final int IMAGE_MAX_SIZE_BYTES = IMAGE_MAX_SIZE_MEGA_BYTES * 1024 * 1024;

    private static final int IMAGE_TURBO_MAX_SIZE_MEGA_BYTES = 300;
    private static final int IMAGE_TURBO_MAX_SIZE_BYTES = IMAGE_TURBO_MAX_SIZE_MEGA_BYTES * 1024 * 1024;

    private static final Set<String> ALLOWED_MIME_TYPES = ImmutableSet.of("image/jpeg", "image/gif", "image/png");
    private static final Set<String> ALLOWED_EXTENSIONS = ImmutableSet.of("jpeg", "gif", "png");

    private final StillageFileInfo stillageFileInfo;
    private final boolean isTurbo;
    private final boolean areAnimatedGifsAllowed;

    public ImageValidator(StillageFileInfo stillageFileInfo, boolean isTurbo, boolean areAnimatedGifsAllowed) {
        super(TankerKeySet.ERROR);
        this.stillageFileInfo = stillageFileInfo;
        this.isTurbo = isTurbo;
        this.areAnimatedGifsAllowed = areAnimatedGifsAllowed;
    }

    public ImageValidator(StillageFileInfo stillageFileInfo, boolean isTurbo) {
        this(stillageFileInfo, isTurbo, false);
    }

    public ImageValidator(StillageFileInfo stillageFileInfo) {
        this(stillageFileInfo, false);
    }

    @Override
    public void validate() {
        if (!isTurbo && stillageFileInfo.getFileSize() > IMAGE_MAX_SIZE_BYTES) {
            addErrorMessage("image-is-too-large", IMAGE_MAX_SIZE_MEGA_BYTES);
        }

        if (isTurbo && stillageFileInfo.getFileSize() > IMAGE_TURBO_MAX_SIZE_BYTES) {
            addErrorMessage("image-is-too-large", IMAGE_TURBO_MAX_SIZE_MEGA_BYTES);
        }

        final String mimeType = stillageFileInfo.getMimeType();
        if (!ALLOWED_MIME_TYPES.contains(mimeType)) {
            addErrorMessage("unsupported-image-format", String.join(", ", ALLOWED_EXTENSIONS));
        }

        // в UserFileService.java разрешаются анимированные гифки, а так же они нужны в инбаннере
        if (!areAnimatedGifsAllowed
                && IMAGE_GIF_VALUE.equalsIgnoreCase(mimeType)
                && !"0.0".equals(stillageFileInfo.getMetadataInfo().get("duration").toString())) {
            addErrorMessage("animated-gif-not-allowed");
        }

        throwValidationErrors();
    }

}
