package ru.yandex.canvas.service;

import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Stopwatch;
import com.google.common.net.HttpHeaders;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.request.body.multipart.ByteArrayPart;
import org.asynchttpclient.request.body.multipart.PartBase;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;
import org.springframework.retry.annotation.Retryable;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.canvas.exceptions.MdsApiException;
import ru.yandex.canvas.model.mds.MdsResponseConverter;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.http.smart.annotations.ResponseHandler;
import ru.yandex.direct.http.smart.converter.ResponseConverterFactory;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.Multipart;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.Part;
import ru.yandex.direct.http.smart.http.Path;
import ru.yandex.inside.mds.MdsMultipartPostResponse;
import ru.yandex.inside.mds.MdsRequestBuilder;
import ru.yandex.misc.io.InputStreamSource;

import static java.util.concurrent.TimeUnit.MILLISECONDS;
import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;

public class MDSService {
    private static final Logger logger = LoggerFactory.getLogger(MDSService.class);
    private static final Marker PERFORMANCE = MarkerFactory.getMarker("PERFORMANCE");

    public static final int TIMEOUT = 60;

    private final String namespace;
    private final MdsRequestBuilder readRequestBuilder;
    private final Api api;

    public MDSService(String mdsWriteUrl, String mdsReadUrl, String namespace, String mdsAuthToken,
                      AsyncHttpClient asyncHttpClient) {
        this.namespace = namespace;
        this.readRequestBuilder = new MdsRequestBuilder(mdsReadUrl, namespace);

        ParallelFetcherFactory parallelFetcherFactory = new ParallelFetcherFactory(asyncHttpClient,
                new FetcherSettings().withRequestTimeout(Duration.ofSeconds(TIMEOUT)));
        this.api = Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("mds_service")
                .withBaseUrl(mdsWriteUrl)
                .withResponseConverterFactory(ResponseConverterFactory.builder()
                        .addConverters(new MdsResponseConverter())
                        .build())
                .addHeaderConfigurator(headers -> headers.add(HttpHeaders.AUTHORIZATION, mdsAuthToken))
                .build()
                .create(Api.class);
    }

    public interface Api {
        @POST("/upload-{namespace}/{upload_id}")
        @Multipart
        @ResponseHandler(parserClass = MdsResponseConverter.class)
        @Headers("Content-type: multipart/form-data")
        Call<MdsMultipartPostResponse> uploadFiles(@Path("namespace") String namespace,
                                                   @Path("upload_id") String uploadId,
                                                   @Part List<PartBase> filesData);
    }

    public static class MDSDir {
        private MdsMultipartPostResponse response;
        private MdsRequestBuilder readRequestBuilder;
        private String dirName;

        public MDSDir(MdsMultipartPostResponse response, MdsRequestBuilder readRequestBuilder, String dirName) {
            this.response = response;
            this.readRequestBuilder = readRequestBuilder;
            this.dirName = dirName;
        }

        public String getURL() {
            return readRequestBuilder.action("get", response.getPosts().get(0).getKey()).toUrl();
        }

        public String getDirUrl() throws MalformedURLException {
            URL url = new URL(getURL());

            return new URL(url.getProtocol(), url.getHost(), url.getPort(),
                    "/get-" + getNameSpace() + "/" + getCoupleId() + "/").toString() + getDirName() + "/";
        }

        public String getNameSpace() {
            return readRequestBuilder.ns;
        }

        public int getCoupleId() {
            return response.getCoupleId();
        }

        public String getDirName() {
            return dirName;
        }
    }

    @Retryable(maxAttempts = 5)
    public MDSDir uploadMultiple(Tuple2List<String, InputStreamSource> files) {
        Stopwatch avatarsUploadStopWatch = Stopwatch.createStarted();

        String dirName = UUID.randomUUID().toString();

        List<PartBase> uploadFilesData = files.stream()
                .map(file -> new ByteArrayPart(dirName + "/" + file.get1(), file.get2().readBytes(),
                        "multipart/form-data", StandardCharsets.UTF_8, file.get1()))
                .collect(Collectors.toList());

        Call<MdsMultipartPostResponse> uploadFilesCall = api.uploadFiles(this.namespace,
                uploadFilesData.get(0).getName(), uploadFilesData);

        logger.info("uploading to mds, url: {}", uploadFilesCall.getRequest().getAHCRequest().getUrl());

        Result<MdsMultipartPostResponse> uploadFilesResult = uploadFilesCall.execute();
        checkResultForErrors(uploadFilesResult, error -> new MdsApiException("Failed to upload file to mds: " + error));

        MdsMultipartPostResponse response = uploadFilesResult.getSuccess();
        logger.info("mds put result: {}", response);

        logger.info(PERFORMANCE, "mds_upload:{}", avatarsUploadStopWatch.elapsed(MILLISECONDS));

        return new MDSDir(response, readRequestBuilder, dirName);
    }
}
