package ru.yandex.canvas.service;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.validation.constraints.NotNull;

import com.fasterxml.jackson.databind.ObjectMapper;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.data.util.Pair;

import ru.yandex.canvas.model.Bundle;
import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.CreativeDocumentBatch;
import ru.yandex.canvas.model.CreativeMigrationResult;
import ru.yandex.canvas.model.MediaSet;
import ru.yandex.canvas.model.elements.Element;
import ru.yandex.canvas.model.elements.ElementType;
import ru.yandex.canvas.model.elements.Fade;
import ru.yandex.canvas.model.elements.Image;
import ru.yandex.canvas.model.elements.Logo;
import ru.yandex.canvas.model.elements.OptionsWithPlaceholder;
import ru.yandex.canvas.model.presets.Preset;
import ru.yandex.canvas.model.presets.PresetItem;
import ru.yandex.canvas.model.presets.PresetSelectionCriteria;

import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.model.Util.deepcopy;

/**
 * Created by pupssman on 10.01.17.
 * <p>
 * Basic migration service.
 * <p>
 * For now handles v.1 -> v.2 migration.
 * Actually, performs "filling up" creative's elements with data from corresponding preset.
 */
public class MigrationService {
    private static final Logger logger = LoggerFactory.getLogger(MigrationService.class);

    private final PresetsService presetsService;
    private final FileService fileService;
    private final ObjectMapper jacksonObjectMapper;  // spring's builtin bean

    public MigrationService(PresetsService presetsService, FileService fileService,
                            ObjectMapper jacksonObjectMapper) {
        this.presetsService = presetsService;
        this.fileService = fileService;
        this.jacksonObjectMapper = jacksonObjectMapper;
    }

    /**
     * Мигрирует батч на новую версию если требуется и подготавливает его для "создания на основе".
     */
    @Nonnull
    public Optional<CreativeDocumentBatch> migrate(@Nonnull CreativeDocumentBatch batch,
                                                   @Nonnull PresetSelectionCriteria selectionCriteria) {
        Optional<Preset> presetO = findPreset(batch, selectionCriteria);
        if (!presetO.isPresent()) {
            return Optional.empty();
        }

        Preset preset = presetO.get();
        int maxPresetVersion = StreamEx.of(preset.getItems())
                .map(CreativeData::getBundle)
                .nonNull()
                .map(Bundle::getVersion)
                .max(Integer::compareTo)
                .orElse(0);

        int maxBatchVersion = StreamEx.of(batch.getItems())
                .map(CreativeDocument::getData)
                .map(CreativeData::getBundle)
                .nonNull()
                .map(Bundle::getVersion)
                .max(Integer::compareTo)
                .orElse(0);

        CreativeDocumentBatch result = maxPresetVersion > maxBatchVersion ? migrate(batch, preset) : batch;

        addPlaceholdersFromPreset(preset, result);
        return Optional.of(result);
    }

    /**
     * Добавляет placeholder'ы к текстовым элементам на промигрированном батче/
     */
    void addPlaceholdersFromPreset(@Nonnull Preset preset, @Nonnull CreativeDocumentBatch batch) {
        // Ввиду уже запланированной переделки формата данных считаем что набор элементов
        // и placeholder'ы к ним у всех элементов батча одинаковый
        Map<Class<? extends Element>, Element> presetElementsByClass = StreamEx
                .of(preset.getItems().get(0).getElements())
                .toMap(Element::getClass, identity());

        for (CreativeDocument creative : batch.getItems()) {
            for (Element batchElement : creative.getData().getElements()) {
                if (batchElement.getOptions() == null) {
                    // No options
                    continue;
                }

                if (!OptionsWithPlaceholder.class.isAssignableFrom(batchElement.getOptions().getClass())) {
                    // No placeholder
                    continue;
                }

                OptionsWithPlaceholder batchElementOptions = (OptionsWithPlaceholder) batchElement.getOptions();

                Element presetElement = presetElementsByClass.get(batchElement.getClass());
                if (presetElement == null) {
                    logger.error("Unable to find preset element for the given creative element");
                    continue;
                }

                OptionsWithPlaceholder presetElementOptions = (OptionsWithPlaceholder) presetElement.getOptions();

                batchElementOptions.setPlaceholder(presetElementOptions.getPlaceholder());
            }
        }
    }

    /**
     * Performs actual migration with given preset.
     * <p>
     * {@link #migrate(CreativeDocumentBatch, PresetSelectionCriteria)} only determines the necessity and
     * possibility for migration and calls this.
     */
    @Nonnull
    CreativeDocumentBatch migrate(@Nonnull CreativeDocumentBatch batch, @Nonnull Preset preset) {
        MigratedCreativeDocumentBatch result = new MigratedCreativeDocumentBatch(batch);

        // migrate existing creatives of the batch
        CreativeMigrationResult creativeMigrationResult = result.getItems().stream()
                .map(creativeDocument -> Pair.of(creativeDocument, findPresetItem(preset, creativeDocument.getData())))
                .filter(pair -> pair.getSecond().isPresent())
                .map(pair -> migrate(pair.getFirst().getData(), pair.getSecond().get()))
                .reduce(CreativeMigrationResult::and)
                .orElse(new CreativeMigrationResult(false));  // no elements migrated -- no changes, duh!

        // get the creative with most of the elements -- as the prototype creative for other sizes
        CreativeDocument prototype = result.getItems().stream()
                .max(Comparator.comparingInt(c -> c.getData().getElements().size()))
                .get();  // should not be empty anyway

        // make all the creative with new sizes based on the prototype
        List<CreativeDocument> newSizes = preset.getItems().stream()
                // only when no creative with same size is already present
                .filter(item -> result.getItems().stream().noneMatch(doc ->
                        item.getWidth().equals(doc.getData().getWidth()) && item.getHeight()
                                .equals(doc.getData().getHeight())))
                .map(item -> resize(prototype, item))
                .collect(toList());

        result.getItems().addAll(newSizes);

        newSizes.forEach(creativeDocument ->
                creativeMigrationResult.addNewSize(MessageFormat.format(
                        "{0,number,#}x{1,number,#}",
                        creativeDocument.getData().getWidth(),
                        creativeDocument.getData().getHeight())));

        result.setMigrationMessage(creativeMigrationResult.getMigrationMessage());

        return result;
    }

    /**
     * Creates new {@link CreativeDocument} using `prototype` for all the data and `presetItem` for reference.
     * <p>
     * NB: copies only the properties required for front-end, take care!
     */
    private CreativeDocument resize(CreativeDocument prototype, PresetItem presetItem) {
        CreativeDocument resized = new CreativeDocument();

        // front-end relies on this so we need to fill it somehow unique
        resized.setId(presetItem.getHeight() * 10000 + presetItem.getWidth());
        resized.setAvailable(false);  // new sizes come in disabled

        resized.setData(deepcopy(jacksonObjectMapper, prototype.getData(), CreativeData.class));
        resized.getData().setWidth(presetItem.getWidth());
        resized.getData().setHeight(presetItem.getHeight());
        resized.withPresetId(presetItem.getId());

        // only reset crop -- geometry is filled up by front-end
        resized.getData().getMediaSets().forEach((s, mediaSet) -> resetCrop(mediaSet));

        Map<String, Element> presetElementMap =
                presetItem.getElements().stream().collect(Collectors.toMap(Element::getType, Function.identity()));
        resized.getData().setElements(resized.getData().getElements().stream()
                .filter(element -> presetElementMap
                        .containsKey(element.getType()))  // leave only the elements in preset
                .peek(element -> {
                    // for Image and Logo we need to copy options from the preset -- they hold geometry
                    // done here same as in migrate(List<Elements> ...)
                    if (element instanceof Image) {
                        ((Image) element).setOptions(
                                deepcopy(jacksonObjectMapper, presetElementMap.get(element.getType()).getOptions(),
                                        Image.Options.class));
                    } else if (element instanceof Logo) {
                        ((Logo) element).setOptions(
                                deepcopy(jacksonObjectMapper, presetElementMap.get(element.getType()).getOptions(),
                                        Logo.Options.class));
                    }
                })
                .collect(toList()));

        return resized;
    }

    /**
     * Migrates a single {@link CreativeData} assuming given {@link PresetItem} as proper prototype.
     * Handles extra actions after element migration performed by {@link #migrate(Collection, Collection)}
     */
    CreativeMigrationResult migrate(@NotNull CreativeData creativeData, @NotNull PresetItem presetItem) {
        CreativeMigrationResult result = new CreativeMigrationResult();

        creativeData.setBundle(presetItem.getBundle());

        ElementsMigrationResult elementsMigrationResult = migrate(creativeData.getElements(), presetItem.getElements());

        if (!elementsMigrationResult.getNewElements().isEmpty()) {
            result.setNewElements(elementsMigrationResult.getNewElements().stream().map(Element::getType)
                    .collect(Collectors.toSet()));

            // add all the mediaSets of new elements to the creativeData
            creativeData.getMediaSets().putAll(
                    elementsMigrationResult.getNewElements().stream().map(Element::getMediaSet)
                            .filter(Objects::nonNull)
                            .collect(Collectors.toMap(
                                    identity(),
                                    set -> presetItem.getMediaSets().get(set)
                            ))
            );
        }

        if (!elementsMigrationResult.getMediaSetsToResetCrop().isEmpty()) {
            result.setHasCropChanges(true);

            creativeData.getMediaSets()
                    .forEach((s, mediaSet) -> resetCrop(mediaSet));
        }

        // NB: handles special cases for migration, AKA kostyl
        elementsMigrationResult.getMergedElements().forEach(element -> {
            if (ElementType.FADE.equals(element.getType())) {
                element.setAvailable(true);

                if ("media-banner_theme_speedy".equals(creativeData.getBundle().getName())) {
                    ((Fade) element).getOptions().setColor(creativeData.getOptions().getBackgroundColor());
                }
            }
        });

        creativeData.setElements(elementsMigrationResult.getMergedElements());

        return result;
    }

    private void resetCrop(MediaSet mediaSet) {
        mediaSet.getItems()
                .forEach(item -> item.getItems()
                        .forEach(subItem -> {
                            subItem.setCroppedFileId(null);
                            fileService.getByIdInternal(subItem.getFileId())
                                    .ifPresent(file -> subItem.setUrl(file.getUrl()));
                        }));
    }

    /**
     * Migrates {@link Element} list of a creative based on supplied preset's {@link Element} list
     * Merge elements that are in the creative and in preset with order of preset's elements:
     * -- iterates over preset's elements
     * -- if a creative's elements contains element with same type, take that
     * -- if not, take preset's element
     *
     * @return {@link ElementsMigrationResult} that contains new Elements collection and extra actions to be performed
     */
    ElementsMigrationResult migrate(@NotNull Collection<Element> creativeElements,
                                    @NotNull Collection<Element> presetElements) {
        Map<String, Element> creativeElementsMap =
                creativeElements.stream().collect(Collectors.toMap(Element::getType, identity()));

        List<Element> newElements = new ArrayList<>();
        List<String> mediaSetsToResetCrop = new ArrayList<>();

        List<Element> mergedElements = presetElements.stream()
                .map(presetElement -> {
                    if (creativeElementsMap.containsKey(presetElement.getType())) {
                        Element oldElement = creativeElementsMap.get(presetElement.getType());

                        // If migrating an Image element, check that it's sizes are same.
                        // If not, flag it's mediaSet for crop reset.
                        if (oldElement instanceof Image) {
                            // TODO: maybe cache these mappings for speed?
                            Image.Options options =
                                    jacksonObjectMapper.convertValue(presetElement.getOptions(), Image.Options.class);
                            Image.Options oldOptions = ((Image) oldElement).getOptions();

                            if (options != null
                                    // FIXME: this assumes width/height are not null
                                    && (!oldOptions.getHeight().equals(options.getHeight())
                                    || !(oldOptions.getWidth().equals(options.getWidth())))) {
                                mediaSetsToResetCrop.add(oldElement.getMediaSet());
                            }
                        }

                        return oldElement;
                    } else {
                        newElements.add(presetElement);
                        Element convertedElement = deepcopy(jacksonObjectMapper, presetElement, Element.class);
                        convertedElement.setAvailable(false); // all new elements are disabled by default
                        return convertedElement;
                    }
                }).collect(toList());

        return new ElementsMigrationResult(newElements, mediaSetsToResetCrop, mergedElements);
    }

    /**
     * Finds proper {@link PresetItem} among {@link Preset} items to migrate creativeData with.
     * It should be of same bundle, size and with version greater than the creativeData
     *
     * @return if any
     */
    Optional<PresetItem> findPresetItem(Preset preset, CreativeData creativeData) {
        return preset.getItems().stream()
                .filter(presetItem -> presetItem.getHeight().equals(creativeData.getHeight())
                        && presetItem.getWidth().equals(creativeData.getWidth())
                )
                .findFirst();
    }

    /**
     * Finds preset for given {batch}.
     * Relies that batch has only same bundle creatives and finds the preset that has all items of this preset.
     */
    Optional<Preset> findPreset(CreativeDocumentBatch batch, PresetSelectionCriteria selectionCriteria) {
        Set<String> batchBundles = batch.getItems().stream().map(creative -> creative.getData().getBundle().getName())
                .collect(Collectors.toSet());

        if (batchBundles.size() != 1) {
            return Optional.empty();  // batch is heterogeneous, cannot find preset for it
        }

        // Временное решение до выхода геопродукта
        return presetsService.getList(selectionCriteria, null, null).stream()
                .filter(preset -> preset.getItems().stream()
                        .allMatch(presetItem -> batchBundles.contains(presetItem.getBundle().getName())))
                .findFirst();
    }

    public static class MigratedCreativeDocumentBatch extends CreativeDocumentBatch {
        private String migrationMessage;

        MigratedCreativeDocumentBatch(CreativeDocumentBatch batch) {
            BeanUtils.copyProperties(batch, this);
        }

        public String getMigrationMessage() {
            return migrationMessage;
        }

        void setMigrationMessage(String migrationMessage) {
            this.migrationMessage = migrationMessage;
        }
    }

    static class ElementsMigrationResult {
        private List<Element> newElements;
        private List<String> mediaSetsToResetCrop;
        private List<Element> mergedElements;

        ElementsMigrationResult(List<Element> newElements, List<String> mediaSetsToResetCrop,
                                List<Element> mergedElements) {
            this.newElements = newElements;
            this.mediaSetsToResetCrop = mediaSetsToResetCrop;
            this.mergedElements = mergedElements;
        }


        List<Element> getNewElements() {
            return newElements;
        }

        List<String> getMediaSetsToResetCrop() {
            return mediaSetsToResetCrop;
        }

        List<Element> getMergedElements() {
            return mergedElements;
        }
    }
}
