package ru.yandex.canvas.service;

import java.net.URI;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.canvas.model.CreativeDocument;
import ru.yandex.canvas.model.html5.Creative;

import static ru.yandex.canvas.CommonConstants.NEW_LIVE_PREVIEW_URL_FOR_COMMON_CREATIVES_ENABLED_FEATURE;
import static ru.yandex.canvas.CommonConstants.NEW_SCREENSHOT_URL_FOR_COMMON_CREATIVES_ENABLED_FEATURE;
import static ru.yandex.canvas.Html5Constants.NEW_LIVE_PREVIEW_URL_FOR_HTML5_CREATIVES_ENABLED_FEATURE;
import static ru.yandex.canvas.Html5Constants.NEW_SCREENSHOT_URL_FOR_HTML5_CREATIVES_ENABLED_FEATURE;

@ParametersAreNonnullByDefault
public class PreviewUrlBuilder {
    private final DirectService directService;
    private final String canvasPreviewHost;
    private final URI webUri;

    public PreviewUrlBuilder(DirectService directService, String canvasPreviewHost, String webUrl) {
        this.directService = directService;
        this.canvasPreviewHost = canvasPreviewHost;
        this.webUri = URI.create(webUrl);
    }

    public String buildPreviewUrl(long clientId, CreativeDocument creative) {
        return buildPreviewUrl(clientId, creative, false);
    }

    /**
     * @return URL to the permanent preview for this creative
     */
    public String buildPreviewUrl(long clientId, CreativeDocument creative, boolean isCompactPreview) {
        boolean isNewLivePreviewUrl = directService.getFeatures(clientId, null)
                .contains(NEW_LIVE_PREVIEW_URL_FOR_COMMON_CREATIVES_ENABLED_FEATURE);

        if (!isNewLivePreviewUrl) {
            return buildOldPreviewUrl(creative, isCompactPreview);
        }

        UriComponentsBuilder uriBuilder = UriComponentsBuilder.newInstance()
                .uri(webUri)
                .pathSegment("creative-preview", "image", creative.getBatchId(), Long.toString(creative.getId()));

        if (isCompactPreview) {
            uriBuilder.queryParam("isCompactPreview", "true");  // set query param
        }

        return uriBuilder.build().toString();
    }

    private String buildOldPreviewUrl(CreativeDocument creative, boolean isCompactPreview) {
        UriComponentsBuilder uriBuilder = UriComponentsBuilder.newInstance()
                .scheme("https")
                .host(canvasPreviewHost)
                .pathSegment("creatives");

        uriBuilder.pathSegment(creative.getBatchId(), Long.toString(creative.getId()), "preview");

        if (isCompactPreview) {
            uriBuilder.queryParam("isCompactPreview", "true");  // set query param
        }

        return uriBuilder.build().toString();
    }

    public String buildHtml5PreviewUrl(long clientId, Creative creative) {
        boolean isNewLivePreviewUrl = directService.getFeatures(clientId, null)
                .contains(NEW_LIVE_PREVIEW_URL_FOR_HTML5_CREATIVES_ENABLED_FEATURE);

        UriComponentsBuilder uriBuilder = UriComponentsBuilder.newInstance();

        if (isNewLivePreviewUrl) {
            uriBuilder.uri(webUri)
                    .pathSegment("creative-preview", "html5", creative.getBatchId(), Long.toString(creative.getId()));
        } else {
            uriBuilder.scheme("https")
                    .host(canvasPreviewHost)
                    .pathSegment("html5", creative.getBatchId(), Long.toString(creative.getId()), "preview");
        }

        return uriBuilder.build().toString();
    }

    public String buildScreenshotUrl(long clientId, CreativeDocument creative) {
        boolean isNewScreenshotUrl = directService.getFeatures(clientId, null)
                .contains(NEW_SCREENSHOT_URL_FOR_COMMON_CREATIVES_ENABLED_FEATURE);

        UriComponentsBuilder screenshotUrl = UriComponentsBuilder.newInstance();
        if (isNewScreenshotUrl) {
            screenshotUrl.uri(webUri)
                    .pathSegment("canvas-api", "creatives", "getScreenshot", Long.toString(creative.getId()));
        } else {
            screenshotUrl.scheme("https").host(canvasPreviewHost)
                    .pathSegment("rest", "creatives", Long.toString(creative.getId()), "getScreenshot");
        }

        return screenshotUrl.build().toString();
    }

    public String buildHtml5ScreenshotUrl(long clientId, Long creativeId) {
        boolean isNewScreenshotUrl = directService.getFeatures(clientId, null)
                .contains(NEW_SCREENSHOT_URL_FOR_HTML5_CREATIVES_ENABLED_FEATURE);

        UriComponentsBuilder screenshotUrl = UriComponentsBuilder.newInstance();
        if (isNewScreenshotUrl) {
            screenshotUrl.uri(webUri)
                    .pathSegment("canvas-api", "html5", "creative", "getScreenshot", creativeId.toString());
        } else {
            screenshotUrl.scheme("https").host(canvasPreviewHost)
                    .pathSegment("html5", "rest", "html5", "creative", creativeId.toString(), "getScreenshot");
        }

        return screenshotUrl.build().toString();
    }

}
