package ru.yandex.canvas.service;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;

import ru.yandex.canvas.exceptions.RTBHostExportException;
import ru.yandex.canvas.service.rtbhost.RtbHostUploadHelper;
import ru.yandex.direct.asynchttp.FetcherSettings;
import ru.yandex.direct.asynchttp.ParallelFetcherFactory;
import ru.yandex.direct.asynchttp.Result;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;
import ru.yandex.direct.bs.dspcreative.service.DspCreativeYtExporter;
import ru.yandex.direct.http.smart.annotations.Json;
import ru.yandex.direct.http.smart.core.Call;
import ru.yandex.direct.http.smart.core.Smart;
import ru.yandex.direct.http.smart.http.Headers;
import ru.yandex.direct.http.smart.http.Multipart;
import ru.yandex.direct.http.smart.http.POST;
import ru.yandex.direct.http.smart.http.Part;

import static ru.yandex.direct.http.smart.error.ErrorUtils.checkResultForErrors;
import static ru.yandex.direct.utils.JsonUtils.toJson;

@ParametersAreNonnullByDefault
public class RTBHostExportService {
    private static final Logger logger = LoggerFactory.getLogger(RTBHostExportService.class);
    private static final String IMPORT_DSP_CREATIVE_ERROR_MESSAGE = "Mapping to import-dsp-creative entry failed";

    private final DspCreativeYtExporter dspCreativeYtExporter;
    // флаг, по которому можно выключить выгрузку, например, в тестах
    private final boolean dspCreativeExportEnabled;
    private final ObjectMapper objectMapper;
    private final String rtbHostBaseUrl;
    private final Api api;

    public RTBHostExportService(DspCreativeYtExporter dspCreativeYtExporter,
                                String rtbHostBaseUrl,
                                boolean dspCreativeExportEnabled,
                                ObjectMapper objectMapper,
                                AsyncHttpClient asyncHttpClient) {
        this.dspCreativeYtExporter = dspCreativeYtExporter;
        this.dspCreativeExportEnabled = dspCreativeExportEnabled;
        this.objectMapper = objectMapper;
        this.rtbHostBaseUrl = rtbHostBaseUrl;

        ParallelFetcherFactory fetcherFactory = new ParallelFetcherFactory(asyncHttpClient, new FetcherSettings());
        this.api = createApi(rtbHostBaseUrl, fetcherFactory);
    }

    private Api createApi(String url, ParallelFetcherFactory parallelFetcherFactory) {
        return Smart.builder()
                .withParallelFetcherFactory(parallelFetcherFactory)
                .withProfileName("rtbhost_service")
                .withBaseUrl(url)
                .addHeaderConfigurator(headers -> headers.add(HttpHeaders.CONTENT_TYPE,
                        MediaType.APPLICATION_JSON_VALUE))
                .build()
                .create(Api.class);
    }

    public interface Api {
        @POST("/export/import-dsp-creative.cgi")
        @Headers("Content-type: multipart/form-data")
        @Multipart
        Call<String> export(@Part("request") @Json List<List<DspCreativeExportEntry>> exportRequest);
    }

    /**
     * Отправить креатив в rtb-host (БК).
     */
    public <T> void exportToRtbHost(List<? extends T> creatives, RtbHostUploadHelper<T> helper) {
        List<DspCreativeExportEntry> dspCreativeEntries = createDspCreativeEntries(creatives, helper);

        exportToRtbHost(dspCreativeEntries);
    }

    /**
     * Filter out invalid creatives and send the rest to RTBHost.
     * <p>
     * Assume success if the POST request returns with 200
     *
     * @return map of invalid creative with their respective error messages
     * @throws RTBHostExportException if export fails
     */
    public <T> Map<T, String> exportValidCreativesToRtbHost(List<T> creatives, RtbHostUploadHelper<T> helper) {
        logger.info("Creating dsp creative entries");
        var dspCreativeEntries = new ArrayList<DspCreativeExportEntry>(creatives.size());
        var invalidCreativesErrorMessages = new HashMap<T, String>(creatives.size());
        for (var creative : creatives) {
            try {
                var dspCreativeExportEntry = helper.toImportDspCreativeEntry(creative, objectMapper);
                dspCreativeEntries.add(dspCreativeExportEntry);
            } catch (Exception e) {
                logger.error(IMPORT_DSP_CREATIVE_ERROR_MESSAGE, e);
                invalidCreativesErrorMessages.put(creative, IMPORT_DSP_CREATIVE_ERROR_MESSAGE + ": " + e);
            }
        }

        if (invalidCreativesErrorMessages.size() == creatives.size()) {
            //no valid creatives to send
            return invalidCreativesErrorMessages;
        }

        exportToRtbHost(dspCreativeEntries);

        return invalidCreativesErrorMessages;
    }

    private void exportToRtbHost(List<DspCreativeExportEntry> dspCreativeEntries) {
        logger.info("Executing POST request to RTB Host, URL: {}, body: {} ",
                rtbHostBaseUrl, toJson(dspCreativeEntries));

        Result<String> exportResult = api.export(List.of(dspCreativeEntries)).execute();
        checkResultForErrors(exportResult, error -> new RTBHostExportException("RTB Host request failed: " + error));

        if (dspCreativeExportEnabled) {
            try {
                dspCreativeYtExporter.export(dspCreativeEntries);
            } catch (Exception e) {
                throw new RTBHostExportException("Failed to export creatives to YT", e);
            }
        }
    }

    public <T> List<DspCreativeExportEntry> createDspCreativeEntries(List<? extends T> creatives,
                                                                     RtbHostUploadHelper<T> helper) {
        try {
            logger.info("Creating dsp creative entries");
            return creatives.stream()
                    .map(creative -> helper.toImportDspCreativeEntry(creative, objectMapper))
                    .collect(Collectors.toList());
        } catch (Exception e) {
            throw new RTBHostExportException("Failed to create dsp creative entries", e);
        }
    }
}
