package ru.yandex.canvas.service;

import java.io.IOException;
import java.util.concurrent.TimeUnit;

import com.google.common.io.Resources;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.CacheControl;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;

import ru.yandex.direct.screenshooter.client.model.ScreenShooterScreenshot;
import ru.yandex.direct.screenshooter.client.service.ScreenShooterClient;

/**
 * @author skirsanov
 */

public class ScreenshooterService {
    private static final Logger logger = LoggerFactory.getLogger(ScreenshooterService.class);
    // temporary link to default image, has to be replaced
    private static final String DEFAULT_SCREENSHOT_PATH = "default_screenshot.png";

    private final ScreenShooterClient screenShooterClient;
    private final AvatarsService avatarsService;
    private final byte[] defaultScreenshot;

    public ScreenshooterService(ScreenShooterClient screenShooterClient,
                                AvatarsService avatarsService) {
        this.screenShooterClient = screenShooterClient;
        this.avatarsService = avatarsService;

        try {
            this.defaultScreenshot = Resources.toByteArray(Resources.getResource(DEFAULT_SCREENSHOT_PATH));
        } catch (IOException e) {
            logger.error("can not load default screenshot", e);
            throw new RuntimeException("can not load default screenshot", e);
        }
    }

    public ScreenShooterScreenshot getScreenshotFromUrl(final String url, final long width, final long height) {
        return screenShooterClient.getScreenshotFromUrl(url, width, height);
    }

    public ScreenShooterScreenshot getScreenshotFromHtml(final String html, final long width, final long height) {
        return screenShooterClient.getScreenshotFromHtml(html, width, height);
    }

    public ResponseEntity<byte[]> prepareScreenshotForResponse(ScreenShooterScreenshot screenshot) {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.IMAGE_PNG);
        headers.add(HttpHeaders.CACHE_CONTROL, CacheControl.maxAge(15, TimeUnit.SECONDS).getHeaderValue());

        final byte[] image;
        if (screenshot.getImageData() != null) {
            image = screenshot.getImageData();
        } else if (screenshot.getIsDone()) {
            image = avatarsService.getImageByUrl(screenshot.getUrl());
        } else {
            image = defaultScreenshot;
        }
        return new ResponseEntity<>(image, headers, HttpStatus.OK);
    }

}
