package ru.yandex.canvas.service;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import org.springframework.data.mongodb.core.FindAndModifyOptions;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.stereotype.Service;

import ru.yandex.canvas.model.Sequence;

@Service
public class SequenceService {

    private static final String CREATIVES_SEQUENCE_ID = "creativeid";

    private static final int INITIAL_CREATIVE_ID = Integer.MAX_VALUE / 2;

    private MongoOperations mongoOperation;

    public SequenceService(MongoOperations mongoOperation) {
        this.mongoOperation = mongoOperation;
        initSequence(CREATIVES_SEQUENCE_ID, INITIAL_CREATIVE_ID);
    }

    /**
     * @param count number of IDs to be reserved
     * @return last generated ID for creative within given count number.
     */
    long getNextCreativeIds(final int count) {
        return getNextIds(CREATIVES_SEQUENCE_ID, count);
    }

    /**
     * @param count of ids to init
     * @return ascending list of ids for new creatives
     */
    public List<Long> getNextCreativeIdsList(final int count) {
        long lastId = getNextIds(CREATIVES_SEQUENCE_ID, count);
        return LongStream.rangeClosed(lastId - count + 1, lastId).boxed().collect(Collectors.toList());
    }

    public List<Sequence> getSequences() {
        return mongoOperation.findAll(Sequence.class);
    }

    private void initSequence(final String sequenceId, int initialId) {
        if (mongoOperation.find(new Query(Criteria.where("id").is(sequenceId)), Sequence.class).isEmpty()) {
            Sequence init = new Sequence();
            init.setId(sequenceId);
            init.setSeq(initialId);
            mongoOperation.insert(init);
        }
    }

    private long getNextIds(final String sequenceId, final int inc) {
        final Update update = new Update();
        update.inc("seq", inc);

        FindAndModifyOptions options = new FindAndModifyOptions();
        options.returnNew(true);

        Sequence seqId = mongoOperation.findAndModify(new Query(Criteria.where("id").is(sequenceId)),
                update, options, Sequence.class);

        if (seqId == null) {
            throw new IllegalStateException("Unable to get id for sequence " + sequenceId);
        }

        return seqId.getSeq();
    }
}
