package ru.yandex.canvas.service;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.model.html5.ProductType;
import ru.yandex.canvas.service.video.VideoCreativeType;

import static ru.yandex.canvas.service.SessionParams.Html5Tag.CPM_PRICE;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.GENERATOR;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.HTML5_CPM_BANNER;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.HTML5_CPM_GEOPRODUCT;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.HTML5_CPM_YNDX_FRONTPAGE;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.PLAYABLE;

public class SessionParams {
    private static final Logger LOGGER = LoggerFactory.getLogger("CANVAS_REQUESTS.log");
    private Long clientId;
    private Set<SessionTag> sessionTags;
    private boolean sharedDataWasReceived = false;
    private List<Long> videoPresetIds = null;
    private SharedDataFilter filter = null;
    private List<Long> presetIds;
    private List<Integer> html5PresetIds;
    private Boolean inBannerFlag;

    public SessionParams(@NotNull HttpServletRequest httpRequest) {
        sessionTags = new HashSet<>();
        String parameter = httpRequest.getParameter("shared_data");

        if (parameter == null) {
            return;
        }

        ObjectMapper objectMapper = new ObjectMapper();

        try {
            SharedData sharedData = objectMapper.readValue(parameter, SharedData.class);
            clientId = sharedData.getClientId();
            sessionTags = parseSharedDataTags(sharedData);
            sharedDataWasReceived = true;
            videoPresetIds = sharedData.getVideoPresetIds();
            filter = sharedData.getFilters();
            presetIds = sharedData.getPresetIds();
            html5PresetIds = sharedData.getHtml5PresetIds();
            inBannerFlag = sharedData.getInBannerFlag();
        } catch (IOException e) {
            LOGGER.info("Invalid shared_data {}", e.getMessage());
        }
    }

    public Long getClientId() {
        return clientId;
    }

    public SessionTag getSessionType() {
        if (sessionTags.isEmpty()) {
            return null;
        }

        return sessionTags.iterator().next();
    }

    public void setSessionType(SessionTag type) {
        sessionTags.clear();
        sessionTags.add(type);
    }

    public Html5Tag getHtml5SessionTag() {
        return getHtml5SessionTag(getSessionType());
    }

    public static Html5Tag getHtml5SessionTag(SessionTag tag) {
        if (tag == null) {
            return HTML5_CPM_BANNER;
        }

        switch (tag) {
            case CPM_BANNER:
                return HTML5_CPM_BANNER;
            case CPM_YNDX_FRONTPAGE:
                return HTML5_CPM_YNDX_FRONTPAGE;
            case CPM_PRICE:
                return CPM_PRICE;
            case PLAYABLE:
                return PLAYABLE;
            case CPM_GEOPRODUCT:
                return HTML5_CPM_GEOPRODUCT;
            case GENERATOR:
                return GENERATOR;
            default:
                throw new BadRequestException(
                        "Current session '" + tag + "' doesn't anticipate any html5 ads");
        }
    }

    private Set<SessionTag> parseSharedDataTags(SharedData sharedData) {
        Set<SessionTag> sessionTags = new HashSet<>();

        if (sharedData == null) {
            sessionTags.add(SessionTag.TEXT);
        } else if (sharedData.getProductType() != null) {
            switch (sharedData.getProductType()) {
                case CPM_GEOPRODUCT:
                    sessionTags.add(SessionTag.CPM_GEOPRODUCT);
                    break;
                case CPM_YNDX_FRONTPAGE:
                    sessionTags.add(SessionTag.CPM_YNDX_FRONTPAGE);
                    break;
                case CPM_PRICE:
                    if (sharedData.getCpmYndxFrontpage() != 0) {//особый вид прайсового видео на главной
                        sessionTags.add(SessionTag.CPM_YNDX_FRONTPAGE);
                    } else {
                        sessionTags.add(SessionTag.CPM_PRICE);
                    }
                    break;
                case PLAYABLE:
                    sessionTags.add(SessionTag.PLAYABLE);
                    break;
                case GENERATOR:
                    sessionTags.add(SessionTag.GENERATOR);
                    break;
                case GENERAL:
                default:
                    sessionTags.add(SessionTag.CPM_BANNER);
                    break;
            }
        } else if (sharedData.getCpmYndxFrontpage() != 0) {
            sessionTags.add(SessionTag.CPM_YNDX_FRONTPAGE);
        } else if (sharedData.cpm != 0 || sharedData.cpmVideo != 0) {
            sessionTags.add(SessionTag.CPM_BANNER);
        } else if (sharedData.nonSkippableCpmVideo != 0 || sharedData.nonSkippableCpm != 0) {
            sessionTags.add(SessionTag.NON_SKIPPABLE_CPM_VIDEO);
        } else if (sharedData.cpmOutdoor != 0) {
            sessionTags.add(SessionTag.CPM_OUTDOOR);
        } else if (sharedData.cpc != 0) {
            sessionTags.add(SessionTag.CPC);
        } else if (sharedData.mobileContent != 0) {
            sessionTags.add(SessionTag.MOBILE_CONTENT);
        } else if (sharedData.getMobileContentVideo() != 0) {
            sessionTags.add(SessionTag.MOBILE_CONTENT_VIDEO);
        } else if (sharedData.cpmIndoor != 0) {
            sessionTags.add(SessionTag.CPM_INDOOR);
        } else if (sharedData.creativeType != null
                && sharedData.creativeType.equalsIgnoreCase("audioAddition") || sharedData.cpmAudio != 0) {
            sessionTags.add(SessionTag.CPM_AUDIO);
        } else if (sharedData.getVideoConstructor() != 0) {
            sessionTags.add(SessionTag.VIDEO_CONSTRUCTOR);
        } else if (sharedData.getCpmGeoproduct() != 0) {
            sessionTags.add(SessionTag.CPM_GEOPRODUCT);
        } else if (sharedData.getCpmGeoPin() != 0) {
            sessionTags.add(SessionTag.CPM_GEO_PIN);
        } else { //e.g. nothing or base=1
            sessionTags.add(SessionTag.TEXT);
        }

        return sessionTags;
    }

    public boolean sessionIs(SessionTag type) {
        return sessionTags.contains(type);
    }

    public boolean isPresent() {
        return sharedDataWasReceived;
    }

    public VideoCreativeType getCreativeType() {
        if (!sharedDataWasReceived) {
            return VideoCreativeType.TEXT;
        }

        if (sessionIs(SessionTag.CPM_BANNER)) {
            return VideoCreativeType.CPM;
        } else if (sessionIs(SessionTag.NON_SKIPPABLE_CPM_VIDEO)) {
            return VideoCreativeType.NON_SKIPPABLE_CPM;
        } else if (sessionIs(SessionTag.CPM_OUTDOOR)) {
            return VideoCreativeType.CPM_OUTDOOR;
        } else if (sessionIs(SessionTag.CPC)) {
            return VideoCreativeType.CPC;
        } else if (sessionIs(SessionTag.MOBILE_CONTENT)) {
            return VideoCreativeType.MOBILE_CONTENT;
        } else if (sessionIs(SessionTag.MOBILE_CONTENT_VIDEO)) {
            return VideoCreativeType.MOBILE_CONTENT_VIDEO;
        } else if (sessionIs(SessionTag.CPM_INDOOR)) {
            return VideoCreativeType.CPM_INDOOR;
        } else if (sessionIs(SessionTag.CPM_AUDIO)) {
            return VideoCreativeType.CPM_AUDIO;
        } else if (sessionIs(SessionTag.VIDEO_CONSTRUCTOR)) {
            return VideoCreativeType.VIDEO_CONSTRUCTOR;
        } else if (sessionIs(SessionTag.CPM_PRICE)) {
            return VideoCreativeType.CPM;
        } else if (sessionIs(SessionTag.CPM_YNDX_FRONTPAGE)) {
            return VideoCreativeType.CPM_YNDX_FRONTPAGE;
        } else {
            return VideoCreativeType.TEXT;
        }
    }

    public void validate() throws BadRequestException {
        if (sessionTags.size() != 1) {
            throw new BadRequestException("Confusing combination of flags was received");
        }
    }

    public List<Long> getVideoPresetIds() {
        return videoPresetIds;
    }

    public SharedDataFilter getFilter() {
        return filter;
    }

    public List<Integer> getHtml5PresetIds() {
        return html5PresetIds;
    }

    public List<Long> getPresetIds() {
        return presetIds;
    }

    public enum SessionTag {
        CPM_BANNER,
        NON_SKIPPABLE_CPM_VIDEO,
        CPM_OUTDOOR,
        CPC,
        MOBILE_CONTENT,
        MOBILE_CONTENT_VIDEO,
        CPM_INDOOR,
        CPM_AUDIO,
        CPM_GEOPRODUCT,
        VIDEO_CONSTRUCTOR,
        CPM_YNDX_FRONTPAGE,
        CPM_PRICE,
        PLAYABLE,
        GENERATOR,
        TEXT,
        CPM_GEO_PIN,
        IN_BANNER,
    }

    public enum Html5Tag {
        HTML5_CPM_BANNER(SessionTag.CPM_BANNER),
        HTML5_CPM_YNDX_FRONTPAGE(SessionTag.CPM_YNDX_FRONTPAGE),
        CPM_PRICE(SessionTag.CPM_PRICE),
        PLAYABLE(SessionTag.PLAYABLE),
        HTML5_CPM_GEOPRODUCT(SessionTag.CPM_GEOPRODUCT),
        GENERATOR(SessionTag.GENERATOR);

        final SessionTag sessionTag;

        Html5Tag(SessionTag sessionTag) {
            this.sessionTag = sessionTag;
        }

        public SessionTag getSessionTag() {
            return sessionTag;
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class SharedDataFilterSize {
        @JsonProperty("w")
        private int w;

        @JsonProperty("h")
        private int h;

        public int getW() {
            return w;
        }

        public int getH() {
            return h;
        }

        public SharedDataFilterSize(int w, int h) {
            this.w = w;
            this.h = h;
        }

        public SharedDataFilterSize() {
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class SharedDataFilter {
        @JsonProperty("sizes")
        private List<SharedDataFilterSize> sizes;

        public List<SharedDataFilterSize> getSizes() {
            return sizes;
        }

        public SharedDataFilter(List<SharedDataFilterSize> sizes) {
            this.sizes = sizes;
        }

        public SharedDataFilter() {
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static class SharedData {
        @JsonProperty("isStretchableHtml5")
        private Boolean isStretchableHtml5;

        @JsonProperty("isCompact")
        private Boolean isCompact;

        @JsonProperty("clientId")
        private Long clientId;

        @JsonProperty("inBanner")
        private Boolean inBannerFlag;

        @JsonProperty("creativeType")
        private String creativeType;

        @JsonProperty("cpc")
        private long cpc;

        @JsonProperty("cpm")
        private long cpm;

        @JsonProperty("non_skippable_cpm_video")
        private long nonSkippableCpmVideo;

        @JsonProperty("non_skippable_cpm")
        private long nonSkippableCpm;

        @JsonProperty("cpm_video")
        private long cpmVideo;

        @JsonProperty("base")
        private long base;

        @JsonProperty("cpm_yndx_frontpage")
        private long cpmYndxFrontpage;

        @JsonProperty("cpm_outdoor")
        private long cpmOutdoor;

        @JsonProperty("mobile_content")
        private long mobileContent;

        @JsonProperty("mobile_content_video")
        private long mobileContentVideo;

        @JsonProperty("cpm_indoor")
        private long cpmIndoor;

        @JsonProperty("cpm_audio")
        private long cpmAudio;

        @JsonProperty("video_constructor")
        private long videoConstructor;

        @JsonProperty("cpm_geoproduct")
        private long cpmGeoproduct;

        @JsonProperty("cpm_geo_pin")
        private long cpmGeoPin;

        @Deprecated
        @JsonProperty("productType")
        private ProductType productType;

        @JsonProperty("videoPresetIds")
        private List<Long> videoPresetIds;

        @JsonProperty("presetIds")
        private List<Long> presetIds;

        @JsonProperty("html5PresetIds")
        private List<Integer> html5PresetIds;

        @JsonProperty("filters")
        private SharedDataFilter filters;

        public Boolean getIsStretchableHtml5() {
            return isStretchableHtml5;
        }

        public Boolean getIsCompact() {
            return isCompact;
        }

        public Long getClientId() {
            return clientId;
        }

        public String getCreativeType() {
            return creativeType;
        }

        public long getCpc() {
            return cpc;
        }

        public long getCpm() {
            return cpm;
        }

        public long getNonSkippableCpmVideo() {
            return nonSkippableCpmVideo;
        }

        public long getMobileContent() {
            return mobileContent;
        }

        public long getCpmOutdoor() {
            return cpmOutdoor;
        }

        public long getCpmIndoor() {
            return cpmIndoor;
        }

        public long getCpmAudio() {
            return cpmAudio;
        }

        public long getCpmVideo() {
            return cpmVideo;
        }

        public long getBase() {
            return base;
        }

        public long getCpmYndxFrontpage() {
            return cpmYndxFrontpage;
        }

        @Deprecated
        public ProductType getProductType() {
            return productType;
        }

        public long getVideoConstructor() {
            return videoConstructor;
        }

        public long getCpmGeoproduct() {
            return cpmGeoproduct;
        }

        public long getCpmGeoPin() {
            return cpmGeoPin;
        }

        public long getMobileContentVideo() {
            return mobileContentVideo;
        }

        public List<Long> getVideoPresetIds() {
            return videoPresetIds;
        }

        public List<Long> getPresetIds() {
            return presetIds;
        }

        public List<Integer> getHtml5PresetIds() {
            return html5PresetIds;
        }

        public Boolean getInBannerFlag() {
            return inBannerFlag;
        }

        public SharedDataFilter getFilters() {
            return filters;
        }
    }

    public Boolean getInBannerFlag() {
        return inBannerFlag;
    }
}
