package ru.yandex.canvas.service;

import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.validation.constraints.NotNull;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.retry.annotation.Retryable;
import org.springframework.web.client.HttpStatusCodeException;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.model.stillage.UploadFileByUrl;

import static ru.yandex.canvas.service.video.HttpUtils.setTestHeaders;

/**
 * Simple HTTP client for Stillage
 * <p>
 * Configured via <b>STILLAGE_URL</b> env var that should point
 * somewhere like <b>https://my-stillage-instance.yandex.ru:8084/api/</b>
 *
 * @author pupssman
 * @see
 * <a href=https://wiki.yandex-team.ru/bannerstorage/stillage/>https://wiki.yandex-team.ru/bannerstorage/stillage/</a>
 */
public class StillageService {
    private static final Logger logger = LoggerFactory.getLogger(StillageService.class);
    private static final String UPLOAD_FILE_PATH = "/v1/files/content/bin";
    private static final String UPLOAD_FILEURL_PATH = "/v1/files/content/externalUrl";
    private static final String UPLOAD_FILEURL_INTERNAL_PATH = "/v1/files/content/url";
    private static final String UPLOAD_SCREENSHOT_PATH = "/v1/files/content/url/screenshot";
    private static final String UPLOAD_VIDEO_WITHOUT_AUDIO_PATH = "/v1/files/content/url/remove_audio";
    private static final String GET_FILE_INFO_PATH = "/v1/files/";
    private static final String COLORWIZ = "colorwiz";
    private static final Set<String> internalHosts = Set.of("s3.mds.yandex.net");
    private final String stillageUrl;
    private final RestTemplate restTemplate;

    public StillageService(final String stillageUrl, RestTemplate restTemplate) {
        this.stillageUrl = stillageUrl;
        this.restTemplate = restTemplate;
    }

    /**
     * Uploads file to Stillage
     *
     * @param fileName some name for a file by content
     * @param body     data of the file
     * @return file metadata from stillage with public URL to access it
     */
    @SuppressWarnings("DefaultAnnotationParam")
    @Retryable(value = RestClientException.class, maxAttempts = 3)
    public StillageFileInfo uploadFile(@NotNull final String fileName, @NotNull final byte[] body) {
        return uploadFile(fileName, body, false, null);
    }

    @SuppressWarnings("DefaultAnnotationParam")
    @Retryable(value = RestClientException.class, maxAttempts = 3)
    public StillageFileInfo uploadFile(@NotNull final String fileName,
                                       @NotNull final byte[] body,
                                       boolean forceUseFilename,
                                       @Nullable String attachmentFilename) {
        final URI requestUri = UriComponentsBuilder.fromHttpUrl(stillageUrl)
                .path(UPLOAD_FILE_PATH)
                .queryParam("fileName", fileName)
                .queryParam("include", COLORWIZ)
                .queryParam("forceUseFilename", forceUseFilename)
                .build()
                .toUri();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_OCTET_STREAM);
        mixXMdsContentTypeHeader(headers, fileName);
        setContentDispositionHeader(headers, attachmentFilename);
        setTestHeaders(headers);

        logger.info("Executing request \"{}\"", requestUri);

        return restTemplate.postForObject(requestUri,
                new HttpEntity<>(body, headers),
                StillageFileInfo.class);
    }

    /**
     * Uploads file to Stillage by URL
     *
     * @param fileName some name for a file
     * @param fileUrl  URL to desired file
     * @return file metadata from stillage with public URL to access it
     */
    @Retryable(value = RestClientException.class, maxAttempts = 3)
    public StillageFileInfo uploadFile(@NotNull final String fileName, @NotNull final URL fileUrl) {
        return uploadFile(fileName, fileUrl, false, null);
    }

    @Retryable(value = RestClientException.class, maxAttempts = 3)
    public StillageFileInfo uploadFile(@NotNull final String fileName,
                                       @NotNull final URL fileUrl,
                                       boolean forceUseFilename,
                                       @Nullable String attachmentFilename) {
        return requestByUrl(getUploadFileUrlPath(fileUrl), fileName, fileUrl.toString(), forceUseFilename,
                attachmentFilename);
    }

    private String getUploadFileUrlPath(@NotNull final URL fileUrl) {
        if (internalHosts.contains(fileUrl.getHost())) {
            return UPLOAD_FILEURL_INTERNAL_PATH;
        }
        return UPLOAD_FILEURL_PATH;
    }

    /**
     * Make screenshot for video from specified URL and upload it using Stillage simultaneously
     * ONLY internal urls!
     *
     * @param fileName some name for a file
     * @param videoUrl URL to desired video
     * @return screenshot metadata from stillage with public URL to access it
     */
    @Retryable(value = RestClientException.class)
    public StillageFileInfo uploadScreenshotForVideoUrlInternal(@NotNull final String fileName,
                                                                @NotNull final String videoUrl) {
        return requestByUrl(UPLOAD_SCREENSHOT_PATH, fileName, videoUrl, false, null);
    }

    /**
     * Upload video to stillage with removing audio from it.
     * ONLY internal urls!
     *
     * @param fileName some name for a file
     * @param videoUrl URL to desired video
     * @return metadata for video with removed audio from stillage with public URL to access it
     */
    @Retryable(value = RestClientException.class)
    public StillageFileInfo uploadVideoWithoutAudioUrlInternal(@NotNull final String fileName,
                                                               @NotNull final String videoUrl) {
        return requestByUrl(UPLOAD_VIDEO_WITHOUT_AUDIO_PATH, fileName, videoUrl, false, null);
    }

    private StillageFileInfo requestByUrl(String stillageEndpointPath,
                                          String fileName,
                                          String fileUrl,
                                          boolean forceUseFilename,
                                          @Nullable String attachmentFilename) {
        final URI requestUri = UriComponentsBuilder.fromHttpUrl(stillageUrl)
                .path(stillageEndpointPath)
                .queryParam("forceUseFilename", forceUseFilename)
                .build()
                .toUri();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        mixXMdsContentTypeHeader(headers, fileName);
        setContentDispositionHeader(headers, attachmentFilename);
        setTestHeaders(headers);

        UploadFileByUrl uploadFileByUrlrequest = new UploadFileByUrl();
        uploadFileByUrlrequest.setFileName(fileName);
        uploadFileByUrlrequest.setFileUrl(fileUrl);

        logger.info("Executing request \"{}\"", requestUri);

        return restTemplate.postForObject(requestUri,
                new HttpEntity<>(uploadFileByUrlrequest, headers),
                StillageFileInfo.class);
    }

    private static void mixXMdsContentTypeHeader(HttpHeaders headers, String fileName) {
        String ext = FilenameUtils.getExtension(fileName);
        if (ext.equals("js")) {
            headers.add("X-Mds-Content-Type", "text/javascript");
        } else if (ext.equals("csv")) {
            headers.add("X-Mds-Content-Type", "application/csv");
        }
    }

    private static void setContentDispositionHeader(HttpHeaders headers, @Nullable String attachmentFilename) {
        if (attachmentFilename != null) {
            String contentDisposition = String.format("attachment; filename*=UTF-8''%s",
                    URLEncoder.encode(attachmentFilename, StandardCharsets.UTF_8).replace("+", "%20"));
            headers.add("X-Mds-Content-Disposition", contentDisposition);
        }
    }

    public Optional<StillageFileInfo> getById(@NotNull final String id) {
        final URI requestUri = UriComponentsBuilder.fromHttpUrl(stillageUrl)
                .path(GET_FILE_INFO_PATH)
                .pathSegment(id)
                .build()
                .toUri();

        logger.info("Executing request \"{}\"", requestUri);

        try {
            return Optional.ofNullable(restTemplate.getForObject(requestUri, StillageFileInfo.class));

            //TODO::make error-handling better
        } catch (HttpStatusCodeException e) {
            if (e.getStatusCode() == HttpStatus.NOT_FOUND) {
                return Optional.empty();
            } else {
                throw e;
            }
        }
    }
}
