package ru.yandex.canvas.service;

import java.text.MessageFormat;
import java.util.List;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.validation.constraints.NotNull;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bayan2.CombineSplitUtil;

import static java.util.ResourceBundle.getBundle;
import static org.springframework.context.i18n.LocaleContextHolder.getLocale;

/**
 * Same KeySets are used in build.gradle file in tanker plugin section.
 * <p>
 * TODO: find a way to sync this enum with keysets mentioned in build.gradle
 *
 * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master">canvas tanker project</a>
 */
public enum TankerKeySet {
    /*
        7 curl -s -H "Expect: " -H "Authorization: OAuth ${TOKEN}" "https://tanker-api.yandex-team.ru/keysets/po/?project-id=canvas&keyset-id=${KEYSET}&language=ru" > ${DIR}/ru/LC_MESSAGES/messages.po
        8 curl -s -H "Expect: " -H "Authorization: OAuth ${TOKEN}" "https://tanker-api.yandex-team.ru/keysets/po/?project-id=canvas&keyset-id=${KEYSET}&language=en" > ${DIR}/en/LC_MESSAGES/messages.po
        9 curl -s -H "Expect: " -H "Authorization: OAuth ${TOKEN}" "https://tanker-api.yandex-team.ru/keysets/po/?project-id=canvas&keyset-id=${KEYSET}&language=tr" > ${DIR}/tr/LC_MESSAGES/messages.po
        10 curl -s -H "Expect: " -H "Authorization: OAuth ${TOKEN}" "https://tanker-api.yandex-team.ru/keysets/po/?project-id=canvas&keyset-id=${KEYSET}&language=uk" > ${DIR}/uk/LC_MESSAGES/messages.po
     */

    HTML5("Html5"),

    VIDEO("Video"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=error">error keyset</a>
     */
    ERROR("error"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=DisclaimerGroups">DisclaimerGroups keyset</a>
     */
    DISCLAIMER_GROUPS("DisclaimerGroups"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=Countries">Countries keyset</a>
     */
    COUNTRIES("Countries"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=Stock">Stock keyset</a>
     */
    STOCK("Stock"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=Presets">Presets keyset</a>
     */
    PRESETS("Presets"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=MediaSetUrl">MediaSetUrl keyset</a>
     */
    MEDIASET_URL("MediaSetUrl"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=MigrationMessages">MigrationMessages keyset</a>
     */
    MIGRATION_MESSAGES("MigrationMessages"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=ideas-list">Ideas list keyset</a>
     */
    IDEAS("Ideas"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=ValidationMessages">Validation messages keyset</a>
     */
    VALIDATION_MESSAGES("ValidationMessages"),

    /**
     * @see <a href="https://tanker.yandex-team.ru/?project=canvas&branch=master&keyset=VideoValidationMessages">Validation messages keyset</a>
     * MessageFormat compatible templates
     */
    VIDEO_VALIDATION_MESSAGES("VideoValidationMessages"),

    OVERLAY_VALIDATION_MESSAGES("OverlayValidationMessages"),

    VIDEO_CONSTRUCTOR_TRANSLATIONS("VideoConstructorTranslations");

    final Logger logger = LoggerFactory.getLogger(TankerKeySet.class);
    private final String resourceLocation;

    TankerKeySet(final String resourceLocation) {
        this.resourceLocation = "localized." + resourceLocation;
    }


    /**
     * @return localized value for given <b>key</b> of current {@link TankerKeySet} according to given <b>locale</b>.
     */
    public String key(final @NotNull String key, final @NotNull Locale locale) {
        return key(key, locale, true);
    }

    /**
     * same as key, but without logging keys without translations, usefull when you only need to translate
     * keys which has translation and it's ok not to.

     * @param key
     * @param locale
     * @return localized value for given <b>key</b> of current {@link TankerKeySet} according to given <b>locale</b>.
     */
    public String keyOmitLogging(final @NotNull String key, final @NotNull Locale locale) {
        return key(key, locale, false);
    }

    private String key(final @NotNull String key, final @NotNull Locale locale, boolean logMissingTranslations) {
        if (key == null) {
            return null;
        }

        ResourceBundle bundle = null;

        try {
            bundle = getBundle(resourceLocation, locale);
        } catch (MissingResourceException e) {
            final StringBuilder messageBuilder = new StringBuilder();

            for (StackTraceElement stackTraceElement : e.getStackTrace()) {
                messageBuilder.append(System.lineSeparator()).append(stackTraceElement.toString());
            }
            logMissingTranslations("Translation bundle '" + toString() + "' not found: "
                    + messageBuilder.toString(), logMissingTranslations);

        }

        if (bundle != null && bundle.containsKey(key) /* && bundle.getLocale().equals(locale) */) {
            List<String> translations = CombineSplitUtil.split(bundle.getString(key)); // un-escape translated string
            if (CollectionUtils.isEmpty(translations)) {
                logMissingTranslations("Empty translations for key " + key + " in bundle " + toString(),
                        logMissingTranslations);
                return key; // empty translations are treated as empty
            }
            final String value = translations.get(0); // get the first inclination from translation
            if (StringUtils.isBlank(value)) {
                logMissingTranslations("Blank line translation for key " + key + " in bundle " + toString(),
                        logMissingTranslations);
                return key;
            }
            return value.trim();
        } else {
            logMissingTranslations("Key " + key + " not found in a bundle " + toString(),
                    logMissingTranslations);
            return key;
        }
    }

    private void logMissingTranslations(String log, boolean shouldLog) {
        if (shouldLog) {
            final StringBuilder messageBuilder = new StringBuilder();

            for(StackTraceElement stackTraceElement : Thread.currentThread().getStackTrace()) {
                messageBuilder.append(System.lineSeparator()).append(stackTraceElement.toString());
            }

            logger.warn(log + messageBuilder.toString());
        }
    }

    /**
     * @return localized value for given <b>key</b> of current {@link TankerKeySet} according to current locale.
     */
    public String key(final @NotNull String key) {  // naming like a god
        return key(key, getLocale());
    }

    public String formattedKey(@NotNull String key, Object... placeholders) {
        return String.format(key(key), placeholders);
    }

    public String interpolate(String msgKey, Object... args) {
        return MessageFormat.format(key(msgKey), args);
    }

}
