package ru.yandex.canvas.service.color;


import static ru.yandex.canvas.service.color.ColorRepresentation.hexToLab;

/**
 * @author skirsanov
 * @see <a href="https://github.com/Gouthamve/ColorSim">CIEDE2000 java implementation</a>
 * @see <a href="https://en.wikipedia.org/wiki/Color_difference#CIEDE2000">CIEDE2000</a>
 */
public final class CIEDE2000ColorDistance {

    private CIEDE2000ColorDistance() {
    }

    public static double compute(final String hexColor1, final String hexColor2) {
        return compute(hexToLab(hexColor1), hexToLab(hexColor2));
    }

    static double compute(double[] lab1, double[] lab2) {
        // Adapted from https://github.com/StanfordHCI/c3

        // parametric factors, use defaults
        double kl = 1, kc = 1, kh = 1;

        double pi = Math.PI,
                l1 = lab1[0], a1 = lab1[1], b1 = lab1[2], cab1 = Math.sqrt(a1 * a1 + b1 * b1),
                l2 = lab2[0], a2 = lab2[1], b2 = lab2[2], cab2 = Math.sqrt(a2 * a2 + b2 * b2),
                cab = 0.5 * (cab1 + cab2),
                g = 0.5 * (1 - Math.sqrt(Math.pow(cab, 7) / (Math.pow(cab, 7) + Math.pow(25, 7)))),
                ap1 = (1 + g) * a1,
                ap2 = (1 + g) * a2,
                cp1 = Math.sqrt(ap1 * ap1 + b1 * b1),
                cp2 = Math.sqrt(ap2 * ap2 + b2 * b2),
                cpp = cp1 * cp2;

        // ensure hue is between 0 and 2pi
        double hp1 = Math.atan2(b1, ap1);
        if (hp1 < 0) {
            hp1 += 2 * pi;
        }
        double hp2 = Math.atan2(b2, ap2);
        if (hp2 < 0) {
            hp2 += 2 * pi;
        }

        double dl = l2 - l1,
                dc = cp2 - cp1,
                dhp = hp2 - hp1;

        if (dhp > +pi) {
            dhp -= 2 * pi;
        }
        if (dhp < -pi) {
            dhp += 2 * pi;
        }
        if (cpp == 0) {
            dhp = 0;
        }

        // Note that the defining equations actually need
        // signed Hue and chroma differences which is different
        // from prior color difference formulae
        double dh = 2 * Math.sqrt(cpp) * Math.sin(dhp / 2);

        // Weighting functions
        double lp = 0.5 * (l1 + l2), cp = 0.5 * (cp1 + cp2);

        // Average Hue Computation
        // This is equivalent to that in the paper but simpler programmatically.
        // Average hue is computed in radians and converted to degrees where needed
        double hp = 0.5 * (hp1 + hp2);
        // Identify positions for which abs hue diff exceeds 180 degrees
        if (Math.abs(hp1 - hp2) > pi) {
            hp -= pi;
        }
        if (hp < 0) {
            hp += 2 * pi;
        }

        // Check if one of the chroma values is zero, in which case set
        // mean hue to the sum which is equivalent to other value
        if (cpp == 0) {
            hp = hp1 + hp2;
        }

        double lpm502 = (lp - 50) * (lp - 50),
                sl = 1 + 0.015 * lpm502 / Math.sqrt(20 + lpm502),
                sc = 1 + 0.045 * cp,
                t = 1 - 0.17 * Math.cos(hp - pi / 6)
                        + 0.24 * Math.cos(2 * hp)
                        + 0.32 * Math.cos(3 * hp + pi / 30)
                        - 0.20 * Math.cos(4 * hp - 63 * pi / 180),
                sh = 1 + 0.015 * cp * t,
                ex = (180 / pi * hp - 275) / 25,
                delthetarad = (30 * pi / 180) * Math.exp(-1 * (ex * ex)),
                rc = 2 * Math.sqrt(Math.pow(cp, 7) / (Math.pow(cp, 7) + Math.pow(25, 7))),
                rt = -1 * Math.sin(2 * delthetarad) * rc;
        dl = dl / (kl * sl);
        dc = dc / (kc * sc);
        dh = dh / (kh * sh);

        // The CIE 00 color difference
        return Math.sqrt(dl * dl + dc * dc + dh * dh + rt * dc * dh);
    }
}
