package ru.yandex.canvas.service.color;

import java.awt.Color;
import java.util.function.Function;

import org.jetbrains.annotations.NotNull;

public class ColorHelper {
    private static final double TO_DARK_SHIFT_THRESHOLD = 0.33d;
    private static final double CONTRAST_MIN_RATIO = 2.5d;
    private static final double ON_FADE_LUMINANCE = 0.9d;

    public static String getBorderColorFrom(@NotNull String cssColor) {
        final Color color = ColorRepresentation.parseCSSColor(cssColor);
        return generateContrastColor(color);
    }

    private static String generateContrastColor(Color color) {
        double relativeLuminance = WGAG20Distance.getLuminance(color);
        if (relativeLuminance > TO_DARK_SHIFT_THRESHOLD) {
            return ColorRepresentation.toCssColor(shiftColor(color, Color::darker));
        }
        return ColorRepresentation.toCssColor(shiftColor(color, Color::brighter));
    }

    public static String adjustColorByLuminance(@NotNull String cssBackgroundColor) {
        final Color backgroundColor = ColorRepresentation.parseCSSColor(cssBackgroundColor);
        return ColorRepresentation.toCssColor(shiftColorsByLuminance(backgroundColor, ON_FADE_LUMINANCE));
    }

    private static Color shiftColorRelativly(Color base, Color relateTo, Function<Color, Color> colorShifter) {
        Color result = base;
        for (int i = 0; i < 15; i++) {
            result = colorShifter.apply(result);
            if (isColorsContrast(result, relateTo)) {
                return result;
            }
        }
        return result;
    }

    private static Color shiftColor(Color base, Function<Color, Color> colorShifter) {
        return shiftColorRelativly(base, base, colorShifter);
    }

    private static boolean isColorsContrast(Color color1, Color color2) {
        return WGAG20Distance.compute(color1, color2) > CONTRAST_MIN_RATIO;
    }

    /**
     * Try to adjust base color to wishedLuminance. Method shifts the color brightness (increasing all color components by
     * the same coefficient)
     *
     * @param base            - color to start from
     * @param wishedLuminance - WGAG20 luminance [0.0 ... 1.0]
     * @return adjusted color
     */
    private static Color shiftColorsByLuminance(Color base, double wishedLuminance) {
        wishedLuminance = Math.max(0.0d, wishedLuminance);
        wishedLuminance = Math.min(1.0d, wishedLuminance);

        double luminance = WGAG20Distance.getLuminance(base);
        if (luminance < 0.055d) {
            // Almost black. Return grey color.
            return shiftColorsByLuminance(Color.white, wishedLuminance);
        }

        // approximate formula of luminance
        double k = Math.pow((wishedLuminance + 0.055d) / (luminance + 0.055d), 1 / 2.4d);

        int redBase = Math.max(base.getRed(), 3);
        int greenBase = Math.max(base.getGreen(), 3);
        int blueBase = Math.max(base.getBlue(), 3);

        double red = Math.min(k * redBase, 255.0d);
        double green = Math.min(k * greenBase, 255.0d);
        double blue = Math.min(k * blueBase, 255.0d);
        return new Color((int) red, (int) green, (int) blue, base.getAlpha());
    }

}
