package ru.yandex.canvas.service.color;

import java.awt.Color;
import java.util.Formatter;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.validation.constraints.NotNull;

/**
 * @author skirsanov
 */
final class ColorRepresentation {
    private static final Pattern CSS_COLOR_REGEXP =
            Pattern.compile("#([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})([0-9A-Fa-f]{2})");


    private static final double XYZEpsilon = 0.008856; // 216/24389
    private static final double XYZKappa = 903.3; // 24389/27
    private static final double[] XYZWhiteReference = new double[]{95.047, 100.000, 108.883};


    private ColorRepresentation() {
    }

    static double[] hexToLab(@NotNull final String hexColorValue) {
        Objects.requireNonNull(hexColorValue);

        double[] rgb = hexToRGB(hexColorValue);
        double[] xyz = RGBtoXYZ(rgb);
        return XYZtoLAB(xyz);
    }

    static double[] hexToRGB(@NotNull final String cssColor) {
        Matcher matcher = CSS_COLOR_REGEXP.matcher(cssColor);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Can't parse css color " + cssColor);
        }

        final int red = Integer.parseInt(matcher.group(1), 16);
        final int green = Integer.parseInt(matcher.group(2), 16);
        final int blue = Integer.parseInt(matcher.group(3), 16);

        return new double[]{
                red,
                green,
                blue
        };
    }

    static Color parseCSSColor(String cssColor) {
        Matcher matcher = CSS_COLOR_REGEXP.matcher(cssColor);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Can't parse css color " + cssColor);
        }
        final String redRaw = matcher.group(1);
        final String greenRaw = matcher.group(2);
        final String blueRaw = matcher.group(3);

        final int red = Integer.parseInt(redRaw, 16);
        final int green = Integer.parseInt(greenRaw, 16);
        final int blue = Integer.parseInt(blueRaw, 16);

        return new Color(red, green, blue);
    }

    static String toCssColor(Color color) {
        return new Formatter().format("#%02x%02x%02x", color.getRed(), color.getGreen(), color.getBlue()).toString();
    }

    private static double PivotRGB(double c) {
        return (c > 0.04045 ? Math.pow((c + 0.055) / 1.055, 2.4) : c / 12.92) * 100.0;
    }

    private static double PivotXYZ(double c) {
        return c > XYZEpsilon ? Math.cbrt(c) : (XYZKappa * c + 16) / 116;
    }

    static double[] RGBtoXYZ(double[] rgb) {
        double[] temp = new double[3];

        for (int i = 0; i < rgb.length; i++) {
            temp[i] = PivotRGB(rgb[i] / 255);
        }

        double[] ret = new double[3];
        // Observer. = 2°, Illuminant = D65
        ret[0] = temp[0] * 0.4124 + temp[1] * 0.3576 + temp[2] * 0.1805;
        ret[1] = temp[0] * 0.2126 + temp[1] * 0.7152 + temp[2] * 0.0722;
        ret[2] = temp[0] * 0.0193 + temp[1] * 0.1192 + temp[2] * 0.9505;

        return ret;
    }

    static double[] XYZtoLAB(double[] xyz) {
        double[] temp = new double[3];

        for (int i = 0; i < 3; i++) {
            temp[i] = PivotXYZ(xyz[i] / XYZWhiteReference[i]);
        }

        double[] result = new double[3];
        result[0] = Math.max(0, 116 * temp[1] - 16);
        result[1] = 500 * (temp[0] - temp[1]);
        result[2] = 200 * (temp[1] - temp[2]);

        return result;
    }
}
