package ru.yandex.canvas.service.direct;

import java.util.List;

import ru.yandex.canvas.Html5Constants;
import ru.yandex.canvas.model.Size;
import ru.yandex.canvas.model.direct.AdditionalData;
import ru.yandex.canvas.model.direct.CreativeUploadComposedFrom;
import ru.yandex.canvas.model.direct.CreativeUploadData;
import ru.yandex.canvas.model.direct.CreativeUploadType;
import ru.yandex.canvas.model.direct.DirectUploadableHelperService;
import ru.yandex.canvas.model.direct.ModerationInfo;
import ru.yandex.canvas.model.direct.YabsData;
import ru.yandex.canvas.model.html5.Creative;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.service.PreviewUrlBuilder;
import ru.yandex.canvas.service.html5.Html5Preset;

import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class Html5CreativeDirectUploadHelper implements DirectUploadableHelperService<Creative> {
    private final PreviewUrlBuilder previewUrlBuilder;

    public Html5CreativeDirectUploadHelper(PreviewUrlBuilder previewUrlBuilder) {
        this.previewUrlBuilder = previewUrlBuilder;
    }


    @Override
    public CreativeUploadData toCreativeUploadData(Creative creative, long clientId) {
        CreativeUploadData creativeUploadData = new CreativeUploadData();

        creativeUploadData.setCreativeId(creative.getId());
        creativeUploadData.setCreativeName(creative.getName());
        creativeUploadData.setCreativeType(CreativeUploadType.HTML5_CREATIVE);
        creativeUploadData.setWidth(creative.getWidth());
        creativeUploadData.setHeight(creative.getHeight());
        creativeUploadData.setArchiveUrl(creative.getArchiveUrl());
        creativeUploadData.setCreativeName(creative.getName());
        creativeUploadData.setDuration(creative.getSource().getVideoDuration());

        Integer presetId = creative.getSource().getPresetId();

        if (presetId == null) {
            presetId = ifNotNull(Html5Constants.HTML5_PRESETS_BY_SIZE.get(new Size(creative.getWidth(),
                    creative.getHeight())), Html5Preset::getId);
        }

        creativeUploadData.setPresetId(presetId);

        ModerationInfo moderationInfo = new ModerationInfo();
        moderationInfo.setContentId(Long.parseLong(creative.getSource().getStillageInfo().getId()));
        moderationInfo.setAdminRejectReason(creative.getAdminRejectReason());
        if (creative.getSource().getVideoUrl() != null) {
            ModerationInfo.Video video = new ModerationInfo.Video();
            video.setUrl(creative.getSource().getVideoUrl());
            moderationInfo.setVideos(List.of(video));
        }
        moderationInfo.setBgrcolor(creative.getSource().getBgrcolor());

        creativeUploadData.setModerationInfo(moderationInfo);
        fillPreviewFields(creativeUploadData, creative, clientId);

        YabsData yabsData = new YabsData()
                .withHtml5("true")
                .withBasePath(creative.getBasePath());

        creativeUploadData.setYabsData(yabsData);

        var sourceImageInfo = creative.getSourceImageInfo();
        if (sourceImageInfo != null) {
            creativeUploadData.setComposedFrom(
                    CreativeUploadComposedFrom.fromStillageInfo(creative.getSource().getSourceImageInfo()));

            if (sourceImageInfo.getMetadataInfo() != null) {
                var additionalData = new AdditionalData();

                var height = sourceImageInfo.getMetadataInfo().get("height");
                var width = sourceImageInfo.getMetadataInfo().get("width");

                if (height != null && width != null) {
                    var originalHeight = (int) height;
                    var originalWidth = (int) width;
                    if (originalHeight != creative.getHeight() && originalWidth != creative.getWidth()) {
                        additionalData.setOriginalHeight(originalHeight);
                        additionalData.setOriginalWidth(originalWidth);
                        creativeUploadData.setAdditionalData(additionalData);
                    }
                }
            }
        }

        creativeUploadData.setIsBrandLift(creative.getIsBrandLift());

        return creativeUploadData;
    }

    /**
     * Заполняет в creativeUploadData информацию о превью креатива
     */
    private void fillPreviewFields(CreativeUploadData creativeUploadData, Creative creative, long clientId) {
        creativeUploadData.setPreviewUrl(creative.getScreenshotUrl());
        creativeUploadData.setLivePreviewUrl(previewUrlBuilder.buildHtml5PreviewUrl(clientId, creative));
        Source source = creative.getSource();
        creativeUploadData.setExpandedPreviewUrl(source.getExpandedHtmlUrl());
        creativeUploadData.getModerationInfo().getHtml().setUrl(source.getHtmlUrl());
        if (source.getExpandedHtmlUrl() != null) {//для расхлопа
            creativeUploadData.setLivePreviewUrl(source.getHtmlUrl());
            //скриншот маленького баннера
            if (source.getScreenshotUrlMainBanner() != null) {
                creativeUploadData.getModerationInfo().getImages().add(
                        new ModerationInfo.Image()
                                .setType("screenshot")
                                .setUrl(source.getScreenshotUrlMainBanner())
                );
            }
        }
    }
}
