package ru.yandex.canvas.service.direct;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.direct.AdditionalData;
import ru.yandex.canvas.model.direct.CreativeUploadData;
import ru.yandex.canvas.model.direct.CreativeUploadType;
import ru.yandex.canvas.model.direct.DirectUploadableHelperService;
import ru.yandex.canvas.model.direct.ModerationInfo;
import ru.yandex.canvas.model.direct.UcCreativeCanvasData;
import ru.yandex.canvas.model.video.Addition;
import ru.yandex.canvas.model.video.addition.options.AdditionElementOptions;
import ru.yandex.canvas.model.video.files.AudioSource;
import ru.yandex.canvas.model.video.files.FileStatus;
import ru.yandex.canvas.model.video.files.Movie;
import ru.yandex.canvas.model.video.files.StreamFormat;
import ru.yandex.canvas.model.video.files.Thumbnail;
import ru.yandex.canvas.model.video.files.VideoSource;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.video.AudioService;
import ru.yandex.canvas.service.video.MovieServiceInterface;
import ru.yandex.canvas.service.video.VideoPreviewUrlBuilder;

import static java.lang.Math.floor;
import static ru.yandex.direct.feature.FeatureName.PERF_CREATIVES_VIDEO_SIZE;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

public class VideoAdditionDirectUploadHelper implements DirectUploadableHelperService<Addition> {
    private static final Logger logger = LoggerFactory.getLogger(VideoAdditionDirectUploadHelper.class);

    private final VideoPreviewUrlBuilder videoPreviewUrlBuilder;
    private final AudioService audioService;
    private final MovieServiceInterface movieService;
    private final DirectService directService;
    private final TankerKeySet videoValidationMessages = TankerKeySet.VIDEO_VALIDATION_MESSAGES;

    public VideoAdditionDirectUploadHelper(VideoPreviewUrlBuilder videoPreviewUrlBuilder, AudioService audioService,
                                           MovieServiceInterface movieService, DirectService directService) {
        this.videoPreviewUrlBuilder = videoPreviewUrlBuilder;
        this.audioService = audioService;
        this.movieService = movieService;
        this.directService = directService;
    }

    @SuppressWarnings("checkstyle:EmptyBlock")
    @Override
    public CreativeUploadData toCreativeUploadData(Addition creative, long clientId) {
        AdditionElementOptions options = creative.findFilesOptions();
        boolean isOverlay = creative.isOverlayAddition();
        boolean cpmAudio = options.getVideoId() == null && !isOverlay;

        CreativeUploadData data = new CreativeUploadData();
        data.setCreativeId(creative.getCreativeId());
        data.setStockCreativeId(creative.getStockCreativeId());
        data.setCreativeType(CreativeUploadType.VIDEO_ADDITION);
        if (creative.getName() != null && creative.getName().length() > 255) {
            data.setCreativeName(creative.getName().substring(0, 255));
        } else {
            data.setCreativeName(creative.getName());
        }
        data.setPreviewUrl(creative.getScreenshotUrl());
        data.setIsBrandLift(creative.getPythia() != null);

        data.setLivePreviewUrl(videoPreviewUrlBuilder.getPreviewUrl(clientId, creative, true));

        if (data.getPreviewUrl() == null) {
            data.setPreviewUrl(data.getLivePreviewUrl());
        }
        data.setPresetId(creative.getPresetId().intValue());

        ModerationInfo moderationInfo = new ModerationInfo();

        ModerationInfo.Html html = new ModerationInfo.Html();
        html.setUrl(videoPreviewUrlBuilder.getPreviewUrl(clientId, creative, true));
        moderationInfo.setHtml(html);

        List<ModerationInfo.Text> texts = creative.getData().getElements().stream()
                .filter(e -> e.getAvailable() && e.getOptions().getText() != null && !e.getOptions().getText()
                        .isEmpty())
                .map(e -> {
                    ModerationInfo.Text t = new ModerationInfo.Text();
                    t.setColor(e.getOptions().getTextColor());
                    t.setType(e.getType().name().toLowerCase());
                    t.setText(e.getOptions().getText());
                    return t;
                }).collect(Collectors.toList());

        moderationInfo.setTexts(texts);

        if (isOverlay) {
            // nothing special yet
        } else if (cpmAudio) {
            AudioSource audioSource = audioService.lookupAudio(options.getAudioId(), creative.getClientId());
            data.setDuration(audioSource.getDuration().doubleValue());
            data.setWidth(0);
            data.setHeight(0);
            data.setLivePreviewUrl(data.getLivePreviewUrl() + "&cpm_audio=1");
            html.setUrl(data.getLivePreviewUrl());
            ModerationInfo.Sound sound = new ModerationInfo.Sound();
            sound.setStockId(audioSource.getStockId());
            sound.setUrl(audioSource.getStillageUrl());
            moderationInfo.setSounds(Collections.singletonList(sound));

            data.setAdditionalData(new AdditionalData().setDuration(data.getDuration()));
        } else {
            Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(), creative.getClientId(),
                    creative.getPresetId());

            if (movie == null) {
                throw new NotFoundException(videoValidationMessages.key("video_file_not_found"));
            }

            data.setDuration(movie.getDuration());
            var features = directService.getFeatures(clientId, null);
            if(movie.getFormats() != null && !movie.getFormats().isEmpty()
                    && features.contains(PERF_CREATIVES_VIDEO_SIZE.getName())) {//проверить фичу
                var maxFormat = StreamEx.of(movie.getFormats())
                        .filter(it -> it.getHeight() != null && it.getWidth() != null)
                        .maxByInt(StreamFormat::getWidth);
                if (maxFormat.isPresent()) {
                    data.setWidth(maxFormat.get().getWidth());
                    data.setHeight(maxFormat.get().getHeight());
                }
            }

            ModerationInfo.Video video = new ModerationInfo.Video();

            video.setStockId(movie.getVideoSource().getStockId());
            video.setUrl(movie.getVideoSource().getStillageUrl());

            moderationInfo.setVideos(Collections.singletonList(video));

            if (movie.getAudioSource() != null && movie.getAudioSource().getStillageUrl() != null) {
                ModerationInfo.Sound sound = new ModerationInfo.Sound();
                sound.setStockId(movie.getAudioSource().getStockId());
                sound.setUrl(movie.getAudioSource().getStillageUrl()); //TODO could be null!
                moderationInfo.setSounds(Collections.singletonList(sound));
            } else {
                moderationInfo.setSounds(Collections.emptyList());
            }
            //these are hard-coded for now. TODO: change when needed
            ModerationInfo.Aspect aspect = new ModerationInfo.Aspect();

            String[] ratio = movie.getVideoSource().getRatio().split(":", 2);

            aspect.setWidth(Long.parseLong(ratio[0]));
            aspect.setHeight(Long.parseLong(ratio[1]));

            moderationInfo.setAspects(Collections.singletonList(aspect));


            AdditionalData additionalData = new AdditionalData()
                    .setDuration(movie.getDuration())
                    .setFormats(
                            movie.getFormats().stream()
                                    .map(e -> new AdditionalData.OutdoorVideoFormat(e)).collect(Collectors.toList()));

            data.setAdditionalData(additionalData);
        }
        data.setModerationInfo(moderationInfo);

        data.setHasPackshot(creative.findFilesOptions().getPackshotId() != null);

        return data;
    }

    public UcCreativeCanvasData toUcCreativeCanvasData(Addition creative) {
        AdditionElementOptions options = creative.findFilesOptions();

        Movie movie = movieService.lookupMovie(options.getVideoId(), options.getAudioId(), creative.getClientId(),
                creative.getPresetId());

        if (movie == null) {
            logger.warn("no movie found for videoId " + options.getVideoId());
            return null;
        }

        UcCreativeCanvasData data = new UcCreativeCanvasData();
        data.setId(creative.getCreativeId());
        data.setSourceUrl(ifNotNull(movie.getVideoSource(), VideoSource::getStillageUrl));
        if (movie.getDuration() != null) {
            data.setDuration((long) floor(movie.getDuration() + 0.5));
        }
        UcCreativeCanvasData.UcCanvasCreativeMeta meta = new UcCreativeCanvasData.UcCanvasCreativeMeta();
        List<StreamFormat> videoFormats = movie.getFormats();
        meta.setThumb(ifNotNull(movie.getVideoSource(), VideoSource::getThumbnail));
        data.setThumbUrl(ifNotNull(ifNotNull(movie.getVideoSource(), VideoSource::getThumbnail),
                Thumbnail::getUrl));

        meta.setFormats(videoFormats);
        meta.setMimeType(ifNotNull(movie.getVideoSource(), VideoSource::getMimeType));
        meta.setStatus(ifNotNull(movie.getStatus(), FileStatus::getValue));
        data.setMeta(meta);
        data.setWidth(ifNotNull(movie.getVideoSource(), VideoSource::getWidth));
        data.setHeight(ifNotNull(movie.getVideoSource(), VideoSource::getHeight));

        return data;
    }
}
