package ru.yandex.canvas.service.html5;

import java.time.LocalDateTime;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.mongodb.client.result.UpdateResult;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Sort;

import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.exceptions.SourceValidationError;
import ru.yandex.canvas.model.Size;
import ru.yandex.canvas.model.html5.Batch;
import ru.yandex.canvas.model.html5.Creative;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.model.validation.Html5SizeValidator;
import ru.yandex.canvas.repository.html5.BatchesRepository;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.PreviewUrlBuilder;
import ru.yandex.canvas.service.SequenceService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.TankerKeySet;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.joining;
import static ru.yandex.canvas.service.SessionParams.SessionTag.CPM_BANNER;

/**
 * Сервис для работы с батчами.
 * <p>
 * Так как теперь появились несколько типов батчей (в частности, обычный батч и батч медийного продукта на Главной),
 * то был введен тип продукта (Общий/Медийный Продукт на Главной). Для сохранения обратной совместимости Общий продукт
 * хранится в базе как отсутствие поля продукт - так, как это было до появления типа продукта. Тип продукта
 * "Медийный Продукт на Главной" хранится как Batch.product_type = 'CPM_YNDX_FRONTPAGE'.
 */
public class Html5BatchesService {
    private static final Logger logger = LoggerFactory.getLogger(Html5BatchesService.class);
    private final SequenceService sequenceService;
    private final BatchesRepository batchesRepository;
    private final PreviewUrlBuilder previewUrlBuilder;
    private final Html5SizeValidator html5SizeValidator;
    private final DirectService directService;

    public Html5BatchesService(SequenceService sequenceService,
                               BatchesRepository batchesRepository,
                               PreviewUrlBuilder previewUrlBuilder, Html5SizeValidator html5SizeValidator,
                               DirectService directService) {
        this.sequenceService = sequenceService;
        this.batchesRepository = batchesRepository;
        this.previewUrlBuilder = previewUrlBuilder;
        this.html5SizeValidator = html5SizeValidator;
        this.directService = directService;
    }

    public Batch createBatchFromSources(Long clientId, String batchName, List<Source> sources,
            SessionParams.Html5Tag productType, Boolean isBrandLift) {
        Set<Size> sizes = StreamEx.of(sources).map(src -> Size.of(src.getWidth(), src.getHeight())).toSet();
        Set<String> features = directService.getFeatures(clientId, null);

        if (!html5SizeValidator.isSizesValid(sizes, features, productType)) {
            throw new SourceValidationError(TankerKeySet.HTML5.formattedKey("unsupported_image_size",
                    html5SizeValidator.validSizesByProductTypeString(features)));
        }

        Batch batch = new Batch();
        batch.setClientId(clientId);
        batch.setDate(LocalDateTime.now());
        batch.setArchive(false);
        batch.setName(batchName);

        if (productType.getSessionTag() != CPM_BANNER) {
            batch.setProductType(productType.getSessionTag());
        }

        List<Long> newIds = sequenceService.getNextCreativeIdsList(sources.size());

        if (logger.isDebugEnabled()) {
            logger.debug("For new creatives got ids: {}",
                    newIds.stream().map(String::valueOf).collect(joining(", ")));
        }

        List<Creative> creatives = StreamEx.zip(newIds, sources, (i, source) -> {
            Creative creative = new Creative();
            creative.setId(i);
            creative.setSource(source); //  entire source, yeah! (с) puppsman ¯\_(ツ)_/¯
            creative.setWidth(source.getWidth());
            creative.setHeight(source.getHeight());
            creative.setScreenshotIsDone(source.getScreenshotIsDone());
            if (!source.getScreenshotIsDone()) {
                creative.setScreenshotUrl(previewUrlBuilder.buildHtml5ScreenshotUrl(clientId, i));
            } else {
                creative.setScreenshotUrl(source.getScreenshotUrl());
            }
            creative.setPreviewUrl(source.getPreviewUrl());
            creative.setArchiveUrl(source.getUrl());
            creative.setSourceImageInfo(source.getSourceImageInfo());
            creative.setBasePath(source.getBasePath());
            creative.setIsBrandLift(isBrandLift);

            return creative;
        }).toList();

        batch.setCreatives(creatives);
        return batchesRepository.createBatch(batch);
    }

    /**
     * Возвращает батч по id
     *
     * @param id          идентификатор батча
     * @param clientId    идентификатор клиента - владельца батча
     * @param productType тип продукта (если CPM_BANNER - то должен передаваться null)
     * @return Batch, соответствующий id И id клиента И типу продукта, иначе исключение NotFoundException
     */
    public Batch getBatch(String id, long clientId, SessionParams.Html5Tag productType) {
        SessionParams.SessionTag converted =
                productType.getSessionTag() != CPM_BANNER ? productType.getSessionTag() : null;
        Batch batch = batchesRepository.getBatchById(clientId, id, converted);

        if (batch == null) {
            throw new NotFoundException();
        }

        return batch;
    }

    /**
     * Обновляет имя батча в репозитории
     *
     * @param id          идентификатор батча
     * @param clientId    идентификатор клиента - владельца батча
     * @param productType тип продукта (если CPM_BANNER - то должен передаваться null)
     * @param name        новое имя батча
     * @return результат операции
     */
    public UpdateResult updateBatchName(String id, long clientId, SessionParams.Html5Tag productType, String name) {
        SessionParams.SessionTag converted =
                productType.getSessionTag() == CPM_BANNER ? null : productType.getSessionTag();
        return batchesRepository.updateBatchName(id, clientId, converted, name);
    }

    /**
     * Архивирует батч в репозитории
     *
     * @param id          идентификатор батча
     * @param clientId    идентификатор клиента - владельца батча
     * @param productType тип продукта (если CPM_BANNER - то должен передаваться null)
     * @return результат операции
     */
    public UpdateResult archiveBatch(String id, long clientId, SessionParams.Html5Tag productType) {
        SessionParams.SessionTag converted =
                productType.getSessionTag() == CPM_BANNER ? null : productType.getSessionTag();

        return batchesRepository.archiveBatch(id, clientId, converted);
    }

    //https://st.yandex-team.ru/CANVAS-1022
    public List<Batch> getBatches(Long clientId, Integer limit, Integer offset,
                                  Sort.Direction sortOrder, Boolean archive, List<String> sizes, String name,
                                  SessionParams.Html5Tag productType) {
        Set<String> sizesSet = new HashSet<>(sizes);

        List<Batch> batches = batchesRepository.getBatchesByQuery(clientId, limit, offset, sortOrder,
                archive, name, productType);

        for (Batch b : batches) {
            b.setAvailable(false);

            for (Creative c : b.getCreatives()) {
                c.setAvailable(sizesSet.isEmpty()
                        || sizesSet.contains(c.getWidth() + "x" + c.getHeight())
                        || sizesSet.contains("0x" + c.getHeight())
                );

                if (c.getAvailable()) {
                    b.setAvailable(true);
                }

                c.setExpandedHtmlUrl(c.getSource().getExpandedHtmlUrl());
                c.setExpanded(c.getExpandedHtmlUrl() != null); //флаг креатива с расхлопом для фротнэнда
                if (c.getSource().getInvalidPath() != null) {
                    c.setValidationError(TankerKeySet.HTML5.formattedKey("has_not_allowed_external_paths",
                            c.getSource().getInvalidPath()));
                }
            }

            b.getCreatives().sort(comparing(e -> !e.getAvailable()));
        }

        return batches;
    }

    /**
     * Отдает общее количество батчей, которое можно найти по ID клиента, имени и флагу архивности.
     * Используется в связке с
     * {@link BatchesRepository#getBatchesByQuery(Long, Integer, Integer, Sort.Direction, Boolean, String)}
     */
    public long getBatchesTotalCount(Long clientId, Boolean archive, String name) {
        return batchesRepository.getBatchesTotalCountByQuery(clientId, archive, name);
    }
}
