package ru.yandex.canvas.service.html5;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.exceptions.BadRequestException;
import ru.yandex.canvas.exceptions.InternalServerError;
import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.exceptions.SourceValidationError;
import ru.yandex.canvas.model.direct.CreativeUploadData;
import ru.yandex.canvas.model.direct.DirectUploadResult;
import ru.yandex.canvas.model.html5.Batch;
import ru.yandex.canvas.model.html5.CheckStatus;
import ru.yandex.canvas.model.html5.Creative;
import ru.yandex.canvas.model.html5.Source;
import ru.yandex.canvas.model.screenshooter.CreativeWithClientId;
import ru.yandex.canvas.repository.html5.BatchesRepository;
import ru.yandex.canvas.service.DirectService;
import ru.yandex.canvas.service.OnCreativeService;
import ru.yandex.canvas.service.RTBHostExportService;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.canvas.service.direct.Html5CreativeDirectUploadHelper;
import ru.yandex.canvas.service.rtbhost.helpers.creative.Html5DspUploadHelper;

import static ru.yandex.canvas.Html5Constants.HTML5_ASYNC_INSPECTION_FEATURE;

public class Html5CreativesService implements OnCreativeService<Creative> {
    private static final Logger logger = LoggerFactory.getLogger(Html5CreativesService.class);

    private final BatchesRepository batchesRepository;
    private final Html5SourcesService sourcesService;
    private final DirectService directService;
    private final RTBHostExportService rtbHostExportService;
    private final Html5DspUploadHelper html5DspUploadHelper;
    private final Html5CreativeDirectUploadHelper html5CreativeDirectUploadHelper;

    public Html5CreativesService(BatchesRepository batchesRepository,
                                 Html5SourcesService sourcesService,
                                 DirectService directService,
                                 RTBHostExportService rtbHostExportService,
                                 Html5DspUploadHelper html5DspUploadHelper,
                                 Html5CreativeDirectUploadHelper html5CreativeDirectUploadHelper) {
        this.batchesRepository = batchesRepository;
        this.sourcesService = sourcesService;
        this.directService = directService;
        this.rtbHostExportService = rtbHostExportService;
        this.html5DspUploadHelper = html5DspUploadHelper;
        this.html5CreativeDirectUploadHelper = html5CreativeDirectUploadHelper;
    }

    public Map<Long, CreativeUploadData> getCreatives(List<Long> parsedIds, Long clientId) {
        List<Batch> batches = batchesRepository.getBatchesByCreativeIds(clientId, parsedIds);

        Set<Long> idsSet = new HashSet<>(parsedIds);

        return batches.stream()
                .flatMap(b -> b.getCreatives().stream().map(c -> enrichCreativeFromBatch(c, b)))
                .filter(c -> idsSet.contains(c.getId()))
                .map(creative -> html5CreativeDirectUploadHelper.toCreativeUploadData(creative, clientId))
                .collect(Collectors.toMap(CreativeUploadData::getCreativeId, Function.identity()));
    }

    protected Creative enrichCreativeFromBatch(Creative creative, Batch batch) {
        creative.setName(batch.getName());
        creative.setBatchId(batch.getId());
        creative.setProductType(batch.getProductType());
        return creative;
    }

    protected void enrichCreativesWithSources(List<Creative> creatives) {
        try {
            sourcesService.downloadZipContent(creatives.stream().map(c -> c.getSource()).collect(Collectors.toList()));
        } catch (IOException | InterruptedException e) {
            logger.error("Failed to download html5 zip content", e);
            throw new InternalServerError("Failed to download html5 zip content");
        }
    }

    public List<Long> uploadHtml5ToDirect(Map<String, List<Long>> batchesIds, Long clientId, Long userId) {

        List<Batch> batches = batchesRepository.getBatchesByIds(clientId, batchesIds.keySet());

        Map<String, Batch> found = batches.stream().
                collect(Collectors.toMap(Batch::getId, Function.identity()));

        Set<String> notFound = batchesIds.keySet().stream()
                .filter(e -> !found.containsKey(e))
                .collect(Collectors.toSet());

        if (notFound.size() > 0) {
            String missingIds = String.join(", ", notFound);
            throw new SourceValidationError("Batches " + missingIds + " are missing");
        }

        List<Creative> creatives = new ArrayList<>();

        for (String batchId : batchesIds.keySet()) {
            Batch dbBatch = found.get(batchId);
            Set<Long> creativeIds = new HashSet<>(batchesIds.get(batchId));

            for (Creative c : dbBatch.getCreatives()) {
                if (creativeIds.contains(c.getId())) {
                    creatives.add(c);
                    enrichCreativeFromBatch(c, dbBatch);
                    creativeIds.remove(c.getId());
                }
            }

            if (creativeIds.size() > 0) {
                String missing = creativeIds.stream().map(s -> s.toString()).collect(Collectors.joining(", "));
                throw new SourceValidationError("Creatives " + missing + " not found in batch " + batchId);
            }
        }

        enrichCreativesWithSources(creatives);

        //валидируем на внешние вызовы, если нужно бросаем SourceValidationError
        if (directService.getFeatures(clientId, userId).contains(HTML5_ASYNC_INSPECTION_FEATURE)) {
            for (Creative creative : creatives) {
                checkForExternalRequestsAndTrowError(creative);
            }
        }

        DirectUploadResult result = directService.sendCreatives(userId, clientId, creatives,
                html5CreativeDirectUploadHelper);
        rtbHostExportService.exportToRtbHost(creatives, html5DspUploadHelper);

        if (result.getUploadResults().size() != creatives.size()
                || result.getUploadResults().stream()
                .anyMatch(e -> !e.getStatus().equals("OK"))
        ) {
            throw new InternalServerError("export failed");
        }

        return creatives.stream().map(Creative::getId).collect(Collectors.toList());
    }

    public void checkForExternalRequestsAndTrowError(Creative creative) {
        var src = creative.getSource();
        if (src.getValidationStatus() == null || src.getValidationStatus() == CheckStatus.VALID) {
            return;
        }
        if (src.getValidationStatus() == CheckStatus.INVALID) {
            var error = TankerKeySet.HTML5.formattedKey("has_not_allowed_external_paths", src.getInvalidPath());
            throw new SourceValidationError(error);
        }
        if (src.getSourceImageInfo() != null) {//Картиночный креатив не нужно валидировать на внешние вызовы
            batchesRepository.updateCreativeValidationStatus(creative.getId(), CheckStatus.VALID, null);
            return;
        }
        try {
            var externalRequests = sourcesService.checkForExternalRequests(src.unzipArchiveContent(),
                    src.getHtmlReplacements(), src.getHtmlFilename());
            if (externalRequests.size() > 0) {
                String paths = String.join(", ", externalRequests);
                batchesRepository.updateCreativeValidationStatus(creative.getId(), CheckStatus.INVALID, paths);
                logger.warn("Not allowed external requests found", new SourceValidationError(paths));
                var error = TankerKeySet.HTML5.formattedKey("has_not_allowed_external_paths", paths);
                throw new SourceValidationError(error);
            }
            //если нет исключения, то обновть результат проверки
            batchesRepository.updateCreativeValidationStatus(creative.getId(), CheckStatus.VALID, null);
        } catch (IOException e) {//ignore zip problem
            logger.warn(e.getMessage(), e);
        }
    }


    /**
     * По списку creativeId возвращает те из них, которые найдены в базе
     *
     * @param creativeIds
     * @return
     */
    @Override
    public List<Long> filterPresent(List<Long> creativeIds) {
        HashSet<Long> idsHashSet = new HashSet<>(creativeIds);
        return getBatchesByCreativeIdsIncludeArchived(creativeIds).stream()
                .flatMap(b -> b.getCreatives().stream()).map(Creative::getId)
                .filter(id -> idsHashSet.contains(id))
                .collect(Collectors.toList());
    }

    @Override
    public List<Creative> fetchByIds(List<Long> creativeIds) {
        List<Batch> batches = getBatchesByCreativeIdsIncludeArchived(creativeIds);
        Set<Long> idsSet = new HashSet<>(creativeIds); // contains optimization
        return batches.stream().flatMap(b -> b.getCreatives().stream())
                .filter(c -> idsSet.contains(c.getId())).collect(Collectors.toList());
    }

    @Override
    public Map<Long, List<Creative>> fetchForTypeByIdsGroupedByClient(List<Long> creativeIds) {
        Set<Long> idsSet = new HashSet<>(creativeIds); // contains optimization
        return getBatchesByCreativeIdsIncludeArchived(creativeIds).stream().collect(Collectors.groupingBy(
                Batch::getClientId,
                Collectors.flatMapping(
                        b -> b.getCreatives().stream()
                                .filter(c -> idsSet.contains(c.getId()))
                                .map(c -> enrichCreativeFromBatch(c, b)),
                        Collectors.toList()
                )
        ));
    }

    @Override
    public Class<Creative> worksOn() {
        return Creative.class;
    }

    public CreativeWithClientId<Creative> getCreativeWithClientIdOrThrow(Long creativeId) {
        Batch batch = batchesRepository.getBatchesByCreativeIdsIncludeArchived(ImmutableList.of(creativeId))
                .stream()
                .findFirst()
                .orElseThrow(NotFoundException::new);

        Creative creative = batch.getCreatives().stream()
                .filter(batchCreative -> batchCreative.getId().longValue() == creativeId)
                .findFirst()
                .orElseThrow(NotFoundException::new);

        enrichCreativeFromBatch(creative, batch);

        return new CreativeWithClientId<>(creative, batch.getClientId());
    }

    protected List<Batch> getBatchesByCreativeIdsIncludeArchived(List<Long> creativeIds) {
        List<Batch> batches = batchesRepository.getBatchesByCreativeIdsIncludeArchived(creativeIds);
        // грязный хак :(
        // Имя креатива используется в DirectService при отправке в Директ, которому поставляются креативы отсюда
        // Красиво сделать можно только поменяв способ хранения в БД (опустив имя с батча на креативы)
        for (Batch batch : batches) {
            for (Creative creative : batch.getCreatives()) {
                enrichCreativeFromBatch(creative, batch);
            }
        }
        return batches;
    }

    /**
     * перестраивает креатив. Перезаливает распакованный zip в MDS и переотправляет в RTBhost
     */
    public void reloadIndex(Creative creative) throws Exception {
        Batch batch = batchesRepository.getBatchesByCreativeIdsIncludeArchived(List.of(creative.getId()))
                .stream().findFirst()
                .orElseThrow(() -> new Exception("batch not found for creative " + creative.getId()));
        if (batch.getProductType() != SessionParams.SessionTag.CPM_YNDX_FRONTPAGE
                && batch.getProductType() != SessionParams.SessionTag.CPM_PRICE) {
            throw new BadRequestException("unsupported product type " + batch.getProductType());
        }
        Source source = creative.getSource();
        String htmlUrl = sourcesService.reuploadIndexHtml(source);//скачать zip перезалить в MDS*/
        //поправить запись sources. В коллекции html5_batches. Поле htmlUrl
        batchesRepository.updateCreativeHtmlUrl(creative.getId(), htmlUrl);

        //переотправить креатив в RTBhost uploadToRtbHost
        rtbHostExportService.exportToRtbHost(List.of(creative), html5DspUploadHelper);
    }
}
