package ru.yandex.canvas.service.html5;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.io.FilenameUtils;
import org.jetbrains.annotations.NotNull;

import ru.yandex.canvas.Html5Constants;
import ru.yandex.canvas.exceptions.SourceValidationError;
import ru.yandex.canvas.model.Size;
import ru.yandex.canvas.model.validation.Html5SizeValidator;
import ru.yandex.canvas.service.SessionParams;
import ru.yandex.canvas.service.TankerKeySet;
import ru.yandex.direct.feature.FeatureName;

import static ru.yandex.canvas.Html5Constants.DEFAULT_MAX_HTML_FILE_SIZE;
import static ru.yandex.canvas.Html5Constants.FRONTPAGE_TAGS;
import static ru.yandex.canvas.Html5Constants.HTML5_PRESETS_BY_SIZE;
import static ru.yandex.canvas.Html5Constants.HTML5_VIDEO_ALLOWED_FEATURE;
import static ru.yandex.canvas.Html5Constants.MAX_CPM_BANNER_FILE_SIZE;
import static ru.yandex.canvas.Html5Constants.MAX_FILE_SIZE_BY_PRESET;
import static ru.yandex.canvas.Html5Constants.MAX_HTML_FILE_SIZE_BY_TAG;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.CPM_PRICE;
import static ru.yandex.canvas.service.SessionParams.Html5Tag.HTML5_CPM_BANNER;

public class Html5Validator {
    private final Html5Zip content;
    private final Html5SizeValidator html5SizeValidator;
    private List<String> errors;


    public Html5Validator(Html5Zip content, Html5SizeValidator html5SizeValidator) {
        this.content = content;
        this.html5SizeValidator = html5SizeValidator;
    }

    private static String getFileExtension(String filename) {
        return FilenameUtils.getExtension(filename).toLowerCase();
    }

    /**
     * Валидирует html5-креатив, загруженный пользователем в виде zip-файла.
     *
     * @param productType
     */
    public Html5ValidationResult validateUserZipCreative(SessionParams.Html5Tag productType, Set<String> features) {
        errors = new ArrayList<>();

        validateZip(content, productType, features);

        if (!errors.isEmpty()) {
            throw new SourceValidationError(errors);
        }

        AdHtmlParser adHtmlParser = validateHtml(content, productType, features);

        if (!errors.isEmpty()) {
            throw new SourceValidationError(errors);
        }

        return new Html5ValidationResult(adHtmlParser);
    }

    public static List<String> getHtmlFileNames(Html5Zip contents) {
        return contents.files().stream()
                .filter(r -> Html5Constants.ALLOWED_HTML_EXTENSIONS.contains(getFileExtension(r)))
                .collect(Collectors.toList());
    }

    /**
     * Валидирует содержимое zip-архива с html5-креативом.
     */
    private void validateZip(Html5Zip contents, SessionParams.Html5Tag productType, Set<String> features) {

        if (contents.files().size() > Html5Constants.MAX_FILES_COUNT) {
            errors.add(TankerKeySet.HTML5.formattedKey("too_much_files_inside_zip", Html5Constants.MAX_FILES_COUNT));
        }

        List<String> htmlFiles = getHtmlFileNames(contents);

        if (htmlFiles.isEmpty()) {
            errors.add(TankerKeySet.HTML5.key("no_html_file_found"));
        }
        if (productType == SessionParams.Html5Tag.CPM_PRICE && htmlFiles.size() > 2) {
            //для расхлопа может быть два
            errors.add(TankerKeySet.HTML5.key("the_html_file_should_be_max_two"));
        }
        if (productType != SessionParams.Html5Tag.CPM_PRICE && htmlFiles.size() > 1) {
            errors.add(TankerKeySet.HTML5.key("the_html_file_should_be_only_one"));
        }

        List<String> incorrectFileNames = contents.files().stream()
                .filter(it -> !checkFileExtension(it, productType, features, htmlFiles))
                .collect(Collectors.toList());

        if (!incorrectFileNames.isEmpty()) {
            String files = String.join(", ", incorrectFileNames);
            errors.add(TankerKeySet.HTML5.formattedKey("not_allowed_file_extension", files));
        }

        List<String> filenamesWithWrongChars = contents.files().stream()
                .filter(r -> r.matches(".*[^-.~\\w/].*"))
                .collect(Collectors.toList());

        if (!filenamesWithWrongChars.isEmpty()) {
            String files = String.join(", ", filenamesWithWrongChars);
            errors.add(TankerKeySet.HTML5.formattedKey("wrong_characters_in_filenames", files));
        }
    }

    @NotNull
    private boolean checkFileExtension(String filename, SessionParams.Html5Tag productType,
                                       Set<String> features, List<String> htmlFiles) {
        if (productType == CPM_PRICE && htmlFiles.size() > 1) {
            //для расхлопа можно видеофайлы грузить
            return Html5Constants.ALLOWED_CPM_PRICE_FILE_EXTENSIONS.contains(getFileExtension(filename));
        }
        if (!FRONTPAGE_TAGS.contains(productType)
                && features != null && features.contains(HTML5_VIDEO_ALLOWED_FEATURE)) {
            Set<String> allowedFileExtensions = ImmutableSet.<String>builder()
                    .addAll(Html5Constants.ALLOWED_FILE_EXTENSIONS)
                    .addAll(Html5Constants.VALID_VIDEO_EXTENSIONS)
                    .build();
            return allowedFileExtensions.contains(getFileExtension(filename));
        }
        return Html5Constants.ALLOWED_FILE_EXTENSIONS.contains(getFileExtension(filename));
    }

    public List<String> getRelativePathsNotInArchive(List<String> relativePaths, String htmlFileName) {
        Path parent = Paths.get(htmlFileName).getParent();
        String htmlDir = parent != null ? parent.normalize() + "/" : "";

        //function to make any file_path be relative to html file location
        //check for relative paths exists in zip archive
        return relativePaths
                .stream()
                //удаляем get параметры file.js?1559506715621 -> file.js
                .map(f -> f.contains("?") ? f.substring(0, f.indexOf("?")) : f)
                .filter(f -> !content.files().contains(htmlDir + f))
                .collect(Collectors.toList());
    }

    protected AdHtmlParser validateHtml(Html5Zip content, SessionParams.Html5Tag productType, Set<String> features) {
        List<String> htmlFileNames = content.files().stream()
                .filter(r -> Html5Constants.ALLOWED_HTML_EXTENSIONS.contains(getFileExtension(r)))
                .collect(Collectors.toList());
        if (htmlFileNames.isEmpty()) {
            return null;
        }

        AdHtmlParser adHtmlParser = new AdHtmlParser(htmlFileNames.stream()
                .map(it -> new AdHtmlParser.HtmlFile(it, content.getFileAsUtf8String(it)))
                .collect(Collectors.toList()),
                productType);
        adHtmlParser.check();

        errors.addAll(adHtmlParser.getErrors());

        Html5Preset preset = HTML5_PRESETS_BY_SIZE.get(Size.of(adHtmlParser.getWidth(), adHtmlParser.getHeight()));
        long maxFileSizeByPreset = preset != null ? MAX_FILE_SIZE_BY_PRESET.get(preset) : 0;

        for (String htmlFileName : htmlFileNames) {
            long htmlFileSize = content.getFileContent(htmlFileName).length;
            long maxAllowedSize = MAX_HTML_FILE_SIZE_BY_TAG.getOrDefault(productType, DEFAULT_MAX_HTML_FILE_SIZE);
            if (productType != null && !FRONTPAGE_TAGS.contains(productType)
                    && features != null && features.contains(HTML5_VIDEO_ALLOWED_FEATURE)) {
                maxAllowedSize = Html5Constants.HTML5_VIDEO_ALLOWED_FILE_SIZE;
            }
            if (productType == HTML5_CPM_BANNER && features != null
                    && features.contains(FeatureName.ALLOW_PROPORTIONALLY_LARGER_IMAGES.getName())) {
                maxAllowedSize = MAX_CPM_BANNER_FILE_SIZE;
            }
            maxAllowedSize = Math.max(maxAllowedSize, maxFileSizeByPreset);
            if (htmlFileSize > maxAllowedSize) {
                errors.add(TankerKeySet.HTML5.formattedKey("file_is_too_big", htmlFileName, maxAllowedSize / 1024));
            }
        }

        Size html5size = Size.of(adHtmlParser.getWidth(), adHtmlParser.getHeight());
        if (!html5SizeValidator.isSizesValid(List.of(html5size), features, productType)
                // TODO DIRECT-94362 временно закрываем размер HTML5 640x134 не картинка, у мобильной морды пока
                // проблемы с его показом, когда они разрешатся, надо будет открыть
                || (Set.of(SessionParams.SessionTag.CPM_YNDX_FRONTPAGE,
                SessionParams.SessionTag.CPM_PRICE).contains(productType.getSessionTag())
                && html5size.equals(Size.of(640, 134)))
        ) {

            errors.add(TankerKeySet.HTML5.formattedKey("unsupported_image_size",
                    html5SizeValidator.validSizesByProductTypeString(features)));
        }

        //function to make any file_path relative to html file location
        //check for relative paths exists in zip archive
        List<String> notFoundRelativePaths =
                getRelativePathsNotInArchive(adHtmlParser.getRelativePaths(), adHtmlParser.getHtmlFileName());
        if (!notFoundRelativePaths.isEmpty()) {
            String paths = String.join(", ", notFoundRelativePaths);
            errors.add(TankerKeySet.HTML5.formattedKey("not_found_in_zip", paths));
        }

        Map<String, String> htmlAndJsContent = new HashMap<>();

        for (String fileName : content.files()) {
            if (fileName.toLowerCase().endsWith(".js") || Html5Constants.ALLOWED_HTML_EXTENSIONS
                    .contains(getFileExtension(fileName))) {
                htmlAndJsContent.put(fileName, content.getFileAsUtf8String(fileName));
            }
        }

        long filesWithClickUrl = htmlAndJsContent.values().stream()
                .filter(c -> c.matches("(?s).*(?:yandexHTML5BannerApi|getClickURLNum).*"))
                .count();

        if (filesWithClickUrl == 0 && productType != CPM_PRICE) {
            errors.add(TankerKeySet.HTML5.key("no_click_urls_found"));
        }

        long imageFilesCount = content.files().stream()
                .filter(r -> Html5Constants.VALID_IMAGE_EXTENSIONS.contains(getFileExtension(r)))
                .count();

        if (imageFilesCount > Html5Constants.IMAGE_COUNT_MERGE_TO_SPRITE * (productType == CPM_PRICE ? 2 : 1)
                && isSSUnchecked(htmlAndJsContent, adHtmlParser.getHtmlFileName())) {
            errors.add(TankerKeySet.HTML5.key("need_to_join_images_into_list_of_sprites"));
        }

        long videoFilesCount = content.files().stream()
                .filter(r -> Html5Constants.VALID_VIDEO_EXTENSIONS.contains(getFileExtension(r)))
                .count();
        if (videoFilesCount > 1) {
            errors.add(TankerKeySet.HTML5.key("the_video_file_should_be_only_one"));
        }

        return adHtmlParser;
    }

    private boolean isSSUnchecked(Map<String, String> htmlAndJsFiles, String htmlFileName) {
        String adobeFlashProGeneratedContent1 = "canvas = document.getElementById(\"canvas\");";
        String adobeFlashProGeneratedContent2 = "canvas = document.getElementById(\"cnvs\");";

        String adobeFlashProSpriteSheetIsDefined = "manifest: [";
        String adobeFlashProSpriteSheetIsCheck = "manifest: []";

        String adobeAnimateCcGeneratedContent1 = "content=\"Adobe_Animate_CC\"";
        String adobeAnimateCcGeneratedContent2 = "AdobeEdge.loadComposition";

        String adobeAnimateSpriteSheetIsDefined = "lib.ssMetadata = [";
        String adobeAnimateSpriteSheetIsUnchecked = "lib.ssMetadata = [];";

        String htmlFileContent = htmlAndJsFiles.get(htmlFileName);

        boolean isAdobeAnimateCC = htmlFileContent.contains(adobeAnimateCcGeneratedContent1)
                || htmlFileContent.contains(adobeAnimateCcGeneratedContent2);

        boolean isAdobeFlashPro = htmlFileContent.contains(adobeFlashProGeneratedContent1)
                || htmlFileContent.contains(adobeFlashProGeneratedContent2);

        boolean isSSDefined = (isAdobeAnimateCC &&
                htmlAndJsFiles.values().stream().anyMatch(f -> f.contains(adobeAnimateSpriteSheetIsDefined)))
                || (isAdobeFlashPro &&
                htmlAndJsFiles.values().stream().anyMatch(f -> f.contains(adobeFlashProSpriteSheetIsDefined)));


        return isSSDefined &&
                (isAdobeAnimateCC &&
                        htmlAndJsFiles.values().stream().anyMatch(f -> f.contains(adobeAnimateSpriteSheetIsUnchecked)))
                ||
                (isAdobeFlashPro &&
                        htmlAndJsFiles.values().stream().noneMatch(f -> f.contains(adobeFlashProSpriteSheetIsCheck)));
    }

    public static class Html5ValidationResult {
        private List<String> pathsForUpload;
        private int width;
        private int height;
        private String html5FileName;
        private String expandedHtmlName;
        private String bgrcolor;

        public Html5ValidationResult(AdHtmlParser adHtmlParser) {
            this.height = adHtmlParser.getHeight();
            this.width = adHtmlParser.getWidth();
            this.pathsForUpload = adHtmlParser.getPatchesForUpload();
            this.html5FileName = adHtmlParser.getHtmlFileName();
            this.expandedHtmlName = adHtmlParser.getExpandedHtmlFileName();
            this.bgrcolor = adHtmlParser.getBgrcolor();
        }

        public Html5ValidationResult(int width, int height, List<String> pathsForUpload, String html5FileName) {
            this.height = height;
            this.width = width;
            this.pathsForUpload = pathsForUpload;
            this.html5FileName = html5FileName;
        }

        public static Html5ValidationResult buildImageHtml5ValidationResult(int width, int height) {
            return new Html5ValidationResult(width, height, Collections.emptyList(), "index.html");
        }

        public int getHeight() {
            return height;
        }

        public String getHtml5FileName() {
            return html5FileName;
        }

        public String getExpandedHtmlName() {
            return expandedHtmlName;
        }

        public List<String> getPathsForUpload() {
            return pathsForUpload;
        }

        public int getWidth() {
            return width;
        }

        public String getBgrcolor() {
            return bgrcolor;
        }
    }
}
