package ru.yandex.canvas.service.html5;

import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.List;
import java.util.Map;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.io.Resources;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.canvas.Html5Constants;
import ru.yandex.canvas.exceptions.InternalServerError;

public class MdsHostedHtml {
    private static final Logger logger = LoggerFactory.getLogger(MdsHostedHtml.class);
    private Document doc;
    private List<List<String>> replacements;
    private Html5SourcesService.VideoUploaded video;
    private HtmlType type;

    public MdsHostedHtml(String htmlFile, List<List<String>> htmlReplacements,
                         Html5SourcesService.VideoUploaded video, HtmlType type) {
        doc = Jsoup.parse(htmlFile);
        replacements = htmlReplacements;
        this.video = video;
        this.type = type;
    }

    public String asHtml() {
        Elements sdks = doc.select("script[src=" + Html5Constants.ADSDK_LIB + "]");
        for (Element e : sdks) {
            e.remove();
        }
        if (type == HtmlType.EXPANDED_BIG || type == HtmlType.SINGLE) {
            String injectionHtml = getInjectionHtml(type);
            String videoUrl = video != null ? video.getUrl() : "";
            Mustache mustache = new DefaultMustacheFactory().compile(new StringReader(injectionHtml), "mdshosted_html.html");
            Map<String, Object> params = Map.of("VIDEO_URL", videoUrl);
            StringWriter writer = new StringWriter();
            mustache.execute(writer, params);
            writer.flush();
            doc.head().children().first().before(writer.toString());
        }
        if (type == HtmlType.EXPANDED_BIG) {
            for (Element v : doc.select("video")) {
                v.attr("pip", "false");
            }
        }

        for (List<String> lst : replacements) {
            String srcLib = lst.get(0);
            String cachedLib = lst.get(1);
            for (Element el : doc.select("script[src=" + srcLib + "]")) {
                el.attr("src", cachedLib);
            }
        }

        doc = new CssMdsPatchUtils(doc).patchCss();

        return doc.toString();
    }

    private String getInjectionHtml(HtmlType type) {
        String resourceName = type == HtmlType.EXPANDED_BIG ? "html5_mds_injection_expanded.html"
                : "html5_mds_injection_single.html";
        try {
            URI presetsUri = Resources.getResource(resourceName).toURI();
            String injectionHtml = new String(Files.readAllBytes(Paths.get(presetsUri)), StandardCharsets.UTF_8);
            return injectionHtml;
        } catch (Exception e) {
            logger.error("filed load injection code " + resourceName, e);
            throw new InternalServerError(e);
        }
    }

    public enum HtmlType {
        EXPANDED_MAIN,//расхлоп основной
        EXPANDED_BIG,//расхлоп раскрытый
        SINGLE,//html5 креатив без расхлопа
    }
}
