package ru.yandex.canvas.service.html5;

import java.net.URI;
import java.util.List;

import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;

import static java.util.stream.Collectors.toList;

/**
 * Simple HTTP client for PhantomJS validator.
 * <p>
 * Configured via <b>PHANTOMJS_URL</b> env var that should point
 * somewhere like <b>https://phantomjs-api.mediaselling.yandex.net</b>
 *
 * @see <a href=https://github.yandex-team.ru/MediaSelling/automoderation-phantomjs-server>PhantomJS moderation srvr</a>
 */
public class PhantomJsCreativesValidator {
    private static final Logger logger = LoggerFactory.getLogger(PhantomJsCreativesValidator.class);
    private static final String CHECK_PATH = "/check";
    private static final String EXTERNAL_REQUEST_MESSAGE_TYPE = "external_request";
    private final String phantomJsUrl;
    private final RestTemplate restTemplate;

    public PhantomJsCreativesValidator(final String phantomJsUrl, RestTemplate restTemplate) {
        this.phantomJsUrl = phantomJsUrl;
        this.restTemplate = restTemplate;
    }

    /**
     * Runs creative's files with PhantomJS and checks for external requests.
     *
     * @param files List of uploaded to stillage creative's files.
     * @return List of external requests' addresses.
     */
    public List<String> checkForExternalRequests(List<Html5SourcesService.FileUploaded> files) {
        if (files.isEmpty()) {
            return List.of();
        }

        final URI requestUri = UriComponentsBuilder.fromHttpUrl(phantomJsUrl)
                .path(CHECK_PATH)
                .build()
                .toUri();

        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        JSONObject phantomJsCheckRequest = new JSONObject();
        phantomJsCheckRequest.put("files", files);
        logger.info("PhantomJS POST request URL: {}, body: {} ", requestUri, phantomJsCheckRequest);

        ResponseEntity<PhantomJsResponse> responseEntity;
        try {
            responseEntity = restTemplate.postForEntity(requestUri,
                    new HttpEntity<>(phantomJsCheckRequest.toString(), headers), PhantomJsResponse.class);
        } catch (RestClientException e) {
            logger.error("PhantomJS validation failed", e);
            return List.of();
        }

        if (responseEntity.getStatusCodeValue() != 200) {
            logger.error("PhantomJS service returned with status code {}. Response: {}",
                    responseEntity.getStatusCodeValue(),
                    responseEntity.getBody());
            return List.of();
        }

        var body = responseEntity.getBody();
        List<LogMessage> logMessages = body != null ? body.getOutput() : List.of();

        return logMessages.stream()
                .filter(message -> message.getType().equals(EXTERNAL_REQUEST_MESSAGE_TYPE))
                .map(LogMessage::getMessage)
                .collect(toList());
    }

    private static class PhantomJsResponse {
        private List<LogMessage> output;
        private String error;

        public List<LogMessage> getOutput() {
            return output;
        }

        public void setOutput(List<LogMessage> output) {
            this.output = output;
        }

        public String getError() {
            return error;
        }

        public void setError(String error) {
            this.error = error;
        }
    }

    private static class LogMessage {
        private int elapsed;
        private String type;
        private String message;
        private String reason;
        private String stacktrace;
        private List<TraceInfo> trace;
        private int status;

        public int getElapsed() {
            return elapsed;
        }

        public void setElapsed(int elapsed) {
            this.elapsed = elapsed;
        }

        public String getType() {
            return type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getMessage() {
            return message;
        }

        public String getReason() {
            return reason;
        }

        public String getStacktrace() {
            return stacktrace;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public void setReason(String reason) {
            this.reason = reason;
        }

        public void setStacktrace(String stacktrace) {
            this.stacktrace = stacktrace;
        }

        public List<TraceInfo> getTrace() {
            return trace;
        }

        public void setTrace(List<TraceInfo> trace) {
            this.trace = trace;
        }

        public int getStatus() {
            return status;
        }

        public void setStatus(int status) {
            this.status = status;
        }
    }

    private static class TraceInfo {
        private String file;
        private String line;
        private String function;

        public String getFile() {
            return file;
        }

        public void setFile(String file) {
            this.file = file;
        }

        public String getLine() {
            return line;
        }

        public void setLine(String line) {
            this.line = line;
        }

        public String getFunction() {
            return function;
        }

        public void setFunction(String function) {
            this.function = function;
        }
    }
}
