package ru.yandex.canvas.service.html5;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.github.mustachejava.DefaultMustacheFactory;
import com.github.mustachejava.Mustache;
import com.google.common.io.Resources;
import org.jsoup.Jsoup;
import org.jsoup.nodes.DataNode;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import ru.yandex.canvas.Html5Constants;
import ru.yandex.canvas.service.SessionParams;

import static ru.yandex.canvas.Html5Constants.HTML5_VIDEO_MUTE_FEATURE;
import static ru.yandex.canvas.Html5Constants.IS_INLINE_EVENT_HANDLERS_FORBIDDEN;

public class RtbhostedHtml {
    private Document doc;
    private List<List<String>> htmlReplacements;
    private final Set<String> features;
    private final SessionParams.Html5Tag productType;
    private boolean rtbHost = false;

    public RtbhostedHtml(String htmlFile, List<List<String>> htmlReplacements, Set<String> features,
                         SessionParams.Html5Tag productType) {
        doc = Jsoup.parse(htmlFile);
        this.htmlReplacements = htmlReplacements;
        this.features = features;
        this.productType = productType;
    }

    private String loadInjectionCode(String auctionDcParams) throws URISyntaxException, IOException {
        String injectionTemplateName = rtbHost ? "html5_rtbhost_injection_code_click_macros.html"
                : "html5_rtbhost_injection_code.html";

        URI presetsUri = Resources.getResource(injectionTemplateName).toURI();
        String injectionHtml = Files.readString(Paths.get(presetsUri));

        Mustache mustache = new DefaultMustacheFactory().compile(new StringReader(injectionHtml), "rtbhosted_html" +
                ".html");

        Map<String, Object> params = new HashMap<>();
        params.put("AUCTION_DC_PARAMS", auctionDcParams);

        StringWriter writer = new StringWriter();

        mustache.execute(writer, params);
        writer.flush();

        return writer.toString();
    }

    public String getPreview(String auctionDcParams) throws IOException, URISyntaxException {
        String injection = loadInjectionCode(auctionDcParams);

        doc.head().children().first().before(new Element("script")
                .appendChild(new DataNode(injection)));

        Elements sdks = doc.select("script[src=" + Html5Constants.ADSDK_LIB + "]");

        for (Element e : sdks) {
            e.remove();
        }

        if (features.contains(HTML5_VIDEO_MUTE_FEATURE)) {
            for (Element v : doc.select("video")) {
                v.attr("muted", "muted");
            }
        }

        for (List<String> lst : htmlReplacements) {
            String srcLib = lst.get(0);
            String cachedLib = lst.get(1);

            Elements libs = doc.select("script[src=" + srcLib + "]");

            for (Element el : libs) {
                el.attr("src", cachedLib);
            }
        }

        //<body onload="..."> переносим в <script> чтобы соблюдать совместимость с CSP  политиками морды
        //многие редкаторы создают баннер с таким обработчиком. Переносим его на лету
        if (productType!= null && IS_INLINE_EVENT_HANDLERS_FORBIDDEN.contains(productType)
                && doc.body() != null && doc.body().hasAttr("onload")) {
            String onloadHendler = doc.body().attr("onload");
            doc.body().removeAttr("onload");
            doc.body().children().last().after(new Element("script")
                    .appendChild(new DataNode(
                            "window.onload = function(){" + onloadHendler + "};"
                    )));
        }

        doc = new CssMdsPatchUtils(doc).patchCss();

        return doc.toString();
    }

    public RtbhostedHtml setRtbHost(boolean rtbHost) {
        this.rtbHost = rtbHost;
        return this;
    }
}
