package ru.yandex.canvas.service.idea;

import java.net.IDN;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Stopwatch;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.Marker;
import org.slf4j.MarkerFactory;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.stereotype.Service;

import ru.yandex.canvas.exceptions.NotFoundException;
import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.DraftCreative;
import ru.yandex.canvas.model.File;
import ru.yandex.canvas.model.IdeaDocument;
import ru.yandex.canvas.model.direct.Privileges;
import ru.yandex.canvas.model.elements.Description;
import ru.yandex.canvas.model.elements.ElementType;
import ru.yandex.canvas.model.ideas.IdeaFilesHelper;
import ru.yandex.canvas.model.presets.Preset;
import ru.yandex.canvas.model.presets.PresetItem;
import ru.yandex.canvas.model.presets.PresetSelectionCriteria;
import ru.yandex.canvas.model.scraper.ScraperData;
import ru.yandex.canvas.model.stillage.StillageFileInfo;
import ru.yandex.canvas.service.AuthService;
import ru.yandex.canvas.service.FileService;
import ru.yandex.canvas.service.PresetsService;
import ru.yandex.canvas.service.color.ColorHelper;
import ru.yandex.canvas.service.idea.modifiers.CreativeColorModifier;
import ru.yandex.canvas.service.idea.modifiers.CreativeColorModifier.ColorKind;
import ru.yandex.canvas.service.idea.modifiers.CreativeModifier;
import ru.yandex.canvas.service.idea.modifiers.ElementBackgroundColorModifier;
import ru.yandex.canvas.service.idea.modifiers.ElementColorModifier;
import ru.yandex.canvas.service.idea.modifiers.ElementConditionalModifier;
import ru.yandex.canvas.service.idea.modifiers.ElementDisableModifier;
import ru.yandex.canvas.service.idea.modifiers.ElementTextModifier;
import ru.yandex.canvas.service.idea.modifiers.ImageModifier;

import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static ru.yandex.canvas.model.Util.deepcopy;
import static ru.yandex.canvas.service.MongoHelper.findByIdQuery;
import static ru.yandex.canvas.service.TankerKeySet.ERROR;
import static ru.yandex.canvas.service.TankerKeySet.IDEAS;

@Service
public class IdeasService {
    private static final Logger logger = LoggerFactory.getLogger(IdeasService.class);
    private static final Marker PERFORMANCE = MarkerFactory.getMarker("PERFORMANCE");

    private static final int FILE_UPLOAD_THREADS = 60; // 10 rps * 3 sec * 2 (2 files per idea and 3 seconds per file)

    private static final String BUTTON_TEXT_KEY = "more-button";
    private static final String SPECIAL_TEXT_KEY = "special-offer-draft";
    private static final String NO_RESULT_ERROR_KEY = "no-result";
    /**
     * {@see {@link IdeasService#generateDraftCreativeId(Preset, PresetItem)}
     */
    private static final int MAX_PRESET_ITEM_ID = 1000000;
    private final Executor uploadExecutor;
    private final PresetsService presetsService;

    private final FileService fileService;

    private final MongoOperations mongoOperations;

    private final AuthService authService;

    private final ObjectMapper objectMapper;

    public IdeasService(PresetsService presetsService, FileService fileService, MongoOperations mongoOperations,
                        AuthService authService, ObjectMapper objectMapper) {
        this.presetsService = presetsService;
        this.fileService = fileService;
        this.mongoOperations = mongoOperations;
        this.authService = authService;
        this.objectMapper = objectMapper;
        uploadExecutor = Executors.newFixedThreadPool(FILE_UPLOAD_THREADS);
    }

    /**
     * Creates and saves the representation of user site main content: colors, images, logos, etc.
     *
     * @param scraperData  - the response of scraper service
     * @param requestedUrl - the url was requested
     * @return - idea document.
     */
    public IdeaDocument createIdea(@NotNull ScraperData scraperData, String requestedUrl, long clientId) {
        authService.requirePermission(Privileges.Permission.IDEA);

        final Stopwatch stopWatch = Stopwatch.createStarted();
        if (scraperData.getImages().isEmpty() || StringUtils.isBlank(scraperData.getTexts().getTitle())) {
            throw new IdeaInsufficientDataException(ERROR.key(NO_RESULT_ERROR_KEY));
        }
        IdeaDocument ideaDocument = new IdeaDocument(scraperData, clientId, new Date(), requestedUrl);
        logger.info(PERFORMANCE, "create_idea:scrap_idea:{}", stopWatch.elapsed(TimeUnit.MILLISECONDS));
        stopWatch.reset().start();
        mongoOperations.insert(ideaDocument);
        logger.info(PERFORMANCE, "create_idea:save_idea:{}", stopWatch.elapsed(TimeUnit.MILLISECONDS));
        return ideaDocument;
    }

    /**
     * @return returns saved idea by db mongo id
     */
    public Optional<IdeaDocument> getIdea(@NotNull String id, long clientId) {
        authService.requirePermission(Privileges.Permission.IDEA);

        return Optional.ofNullable(mongoOperations.findOne(findByIdQuery(id, clientId),
                IdeaDocument.class));
    }


    /**
     * Makes draft creatives from idea and first size of each presets. Do not saves images and drafts to db.
     *
     * @param ideaDocument - idea document
     * @param selectionCriteria
     * @return list of draft creatives created
     */
    public List<DraftCreative> previewDraftCreatives(@NotNull IdeaDocument ideaDocument,
                                                     @NotNull PresetSelectionCriteria selectionCriteria) {
        authService.requirePermission(Privileges.Permission.IDEA);

        List<DraftCreative> result = new ArrayList<>();

        Map<String, File> scrapedFilesStubs = generateFakeFiles(ideaDocument);

        List<DraftCreative> creativePresetsDrafts;

        final AtomicInteger idCounter = new AtomicInteger(0);

        // Временное решение до выхода геопродукта
        creativePresetsDrafts = presetsService.getList(selectionCriteria, null, null).stream()
                .filter(e -> !PresetsService.IN_BANNER_PRESET_IDS.contains(e.getId()))
                .map(preset -> new DraftCreative(preset.getId(),
                        deepcopy(objectMapper, preset.getItems().get(0), CreativeData.class),
                        generateDraftCreativeId(preset, preset.getItems().get(0))))
                .collect(toList());

        creativePresetsDrafts
                .forEach(draft -> applyModifiers(draft, getModifiersBasedOnScraper(ideaDocument, scrapedFilesStubs)));

        creativePresetsDrafts.forEach(this::filterUnavailableElements);

        creativePresetsDrafts.forEach(creative -> creative.setId(idCounter.getAndIncrement()));

        result.addAll(creativePresetsDrafts);

        return result;
    }

    public List<DraftCreative> generatePresetDraftCreatives(@NotNull IdeaDocument idea, @NotNull Integer presetId,
                                                            @NotNull PresetSelectionCriteria selectionCriteria) {
        authService.requirePermission(Privileges.Permission.IDEA);
        authService.requirePermission(Privileges.Permission.CREATIVE_CREATE);
        authService.requirePermission(Privileges.Permission.CREATIVE_GET);

        // Временное решение до выхода геопродукта
        final Preset preset = presetsService.getList(selectionCriteria, null, null).stream()
                .filter(e -> !PresetsService.IN_BANNER_PRESET_IDS.contains(e.getId()))
                .filter(p -> Objects.equals(p.getId(), presetId)).findFirst()
                .orElseThrow(NotFoundException::new);

        final Stopwatch stopWatch = Stopwatch.createStarted();
        List<DraftCreative> creativePresetsDrafts = preset.getItems().stream()
                .map(presetItem -> new DraftCreative(preset.getId(),
                        deepcopy(objectMapper, presetItem, CreativeData.class),
                        generateDraftCreativeId(preset, presetItem)))
                .collect(toList());
        logger.info(PERFORMANCE, "generate_creatives:presets_to_creatives:{}",
                stopWatch.elapsed(TimeUnit.MILLISECONDS));

        stopWatch.reset().start();
        Map<String, File> scrapedFiles = uploadFiles(idea);
        logger.info(PERFORMANCE, "generate_creatives:gen_modifiers:{}", stopWatch.elapsed(TimeUnit.MILLISECONDS));

        stopWatch.reset().start();
        creativePresetsDrafts.forEach(draft -> applyModifiers(draft, getModifiersBasedOnScraper(idea, scrapedFiles)));
        logger.info(PERFORMANCE, "generate_creatives:apply_modifiers:{}", stopWatch.elapsed(TimeUnit.MILLISECONDS));

        return creativePresetsDrafts;
    }

    private Map<String, File> generateFakeFiles(@NotNull IdeaDocument idea) {
        return Stream.of(
                idea.getScraperData().getLogos().stream().findFirst(),
                idea.getScraperData().getImages().stream().findFirst()
        ).filter(Optional::isPresent)
                .map(Optional::get) // TODO::replace with Java 9's Optional::stream
                .map(stillageFile -> IdeaFilesHelper.generateFakeFile(idea, stillageFile))
                .collect(Collectors.toMap(File::getStillageFileId, Function.identity(), (file1, file2) -> file1));
    }

    /**
     * @return hacky "unique" (among all our presets) preset item ID
     */
    private Integer generateDraftCreativeId(Preset preset, PresetItem presetItem) {
        return preset.getId() * MAX_PRESET_ITEM_ID + presetItem.getId();
    }

    private Map<String, File> uploadFiles(@NotNull IdeaDocument idea) {
        final List<CompletableFuture<File>> futures = Stream.of(
                idea.getScraperData().getLogos().stream().findFirst(),
                idea.getScraperData().getImages().stream().findFirst()
        )
                .filter(Optional::isPresent).map(Optional::get) // TODO::replace with Java 9's Optional::stream
                .map(StillageFileInfo::getId)
                .map(id -> CompletableFuture.supplyAsync(
                        () -> fileService.uploadIdeaFileInternal(id, idea.getId(), idea.getClientId()),
                        uploadExecutor)
                )
                .collect(toList());

        return CompletableFuture.allOf(futures.toArray(new CompletableFuture[futures.size()]))
                .thenApply(v -> futures.stream().map(CompletableFuture::join) // IGNORE-BAD-JOIN DIRECT-149116
                        .collect(Collectors.toMap(File::getStillageFileId, identity())))
                .join(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    private List<CreativeModifier> getModifiersBasedOnScraper(@NotNull IdeaDocument idea,
                                                              Map<String, File> uploadedFiles) {
        final List<CreativeModifier> modifiers = new ArrayList<>();
        ScraperData.ScraperColors colorScheme = idea.getScraperData().getColors();

        modifiers.add(new ElementTextModifier(ImmutableSet.of(ElementType.DOMAIN),
                IDN.toUnicode(idea.getScraperData().getHostname())));

        modifiers.add(new ElementTextModifier(ImmutableSet.of(ElementType.BUTTON), IDEAS.key(BUTTON_TEXT_KEY)));

        modifiers.add(new ElementTextModifier(ImmutableSet.of(ElementType.SPECIAL), IDEAS.key(SPECIAL_TEXT_KEY)));

        modifiers.addAll(generateLogoModifiers(idea, uploadedFiles));

        //remove disclaimer, age restriction, legal, fade by default
        modifiers.add(new ElementDisableModifier(ElementType.DISCLAIMER));
        modifiers.add(new ElementDisableModifier(ElementType.LEGAL));
        modifiers.add(new ElementDisableModifier(ElementType.AGE_RESTRICTION));

        modifiers.add(ElementConditionalModifier
                .apply(new ElementDisableModifier(ElementType.FADE))
                .ifPresetIdNotIn(1));

        if (!idea.getScraperData().getImages().isEmpty()) {
            modifiers.addAll(generateImageModifiers(idea, uploadedFiles));
        }


        if (StringUtils.isNotBlank(colorScheme.getBorder())) {
            modifiers.add(new CreativeColorModifier(ColorKind.Border,
                    ColorHelper.getBorderColorFrom(colorScheme.getBackground())));
        }

        if (StringUtils.isNotBlank(colorScheme.getButton())) {
            modifiers.add(
                    new ElementBackgroundColorModifier(ImmutableSet.of(ElementType.BUTTON, ElementType.SPECIAL),
                            colorScheme.getButton()));
        }

        if (StringUtils.isNotBlank(colorScheme.getText())) {
            modifiers.add(ElementConditionalModifier.apply(
                    new ElementColorModifier(
                            ImmutableSet.of(
                                    ElementType.DESCRIPTION,
                                    ElementType.SPECIAL,
                                    ElementType.DISCLAIMER,
                                    ElementType.AGE_RESTRICTION,
                                    ElementType.HEADLINE,
                                    ElementType.DOMAIN
                            ),
                            colorScheme.getText()
                    )).ifPresetIdNotIn(1));
        }

        if (StringUtils.isNotBlank(colorScheme.getBackground())) {
            modifiers.add(new CreativeColorModifier(ColorKind.Background, colorScheme.getBackground()));
            modifiers.add(ElementConditionalModifier.apply(new ElementColorModifier(
                    ImmutableSet.of(ElementType.DESCRIPTION,
                            ElementType.SPECIAL,
                            ElementType.DISCLAIMER,
                            ElementType.AGE_RESTRICTION,
                            ElementType.HEADLINE,
                            ElementType.DOMAIN),
                    ColorHelper.adjustColorByLuminance(colorScheme.getBackground())))
                    .ifPresetIdIn(1));
        }

        if (StringUtils.isNotBlank(colorScheme.getButtonText())) {
            modifiers.add(
                    new ElementColorModifier(ImmutableSet.of(ElementType.BUTTON), colorScheme.getButtonText()));
        }

        if (StringUtils.isNotBlank(idea.getScraperData().getTexts().getTitle())) {
            modifiers.add(new ElementTextModifier(ImmutableSet.of(ElementType.HEADLINE),
                    idea.getScraperData().getTexts().getTitle()));
        }
        String description = idea.getScraperData().getTexts().getDescription();
        if (StringUtils.isNotBlank(description) &&
                StringUtils.length(description) <= Description.DESCRIPTION_MAX_LENGTH) {
            modifiers.add(new ElementTextModifier(ImmutableSet.of(ElementType.DESCRIPTION), description));
        } else {
            modifiers.add(new ElementDisableModifier(ElementType.DESCRIPTION));
        }
        return modifiers;
    }

    private void filterUnavailableElements(@NotNull DraftCreative draftCreative) {
        draftCreative.getData().getElements().removeIf(element -> !element.getAvailable());
    }

    private List<CreativeModifier> generateImageModifiers(@NotNull IdeaDocument idea, Map<String, File> uploadedFiles) {
        StillageFileInfo image = idea.getScraperData().getImages().get(0);

        if (!uploadedFiles.containsKey(image.getId())) {
            throw new IllegalStateException("scraper image required to be present in uploaded files");
        }
        return singletonList(new ImageModifier(uploadedFiles.get(image.getId()), ElementType.IMAGE));
    }

    private List<CreativeModifier> generateLogoModifiers(@NotNull IdeaDocument idea, Map<String, File> uploadedFiles) {
        if (idea.getScraperData().getLogos().isEmpty()) {
            return Collections.singletonList(new ElementDisableModifier(ElementType.LOGO));
        }
        StillageFileInfo logo = idea.getScraperData().getLogos().get(0);

        if (!uploadedFiles.containsKey(logo.getId())) {
            throw new IllegalStateException("scraper logo required to be present in uploaded files");
        }
        return Collections.singletonList(new ImageModifier(uploadedFiles.get(logo.getId()), ElementType.LOGO));
    }

    private void applyModifiers(@NotNull DraftCreative creative, @NotNull List<CreativeModifier> modifiers) {
        modifiers.forEach(modifier -> modifier.modify(creative));
    }
}
