package ru.yandex.canvas.service.idea.modifiers;

import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.canvas.model.DraftCreative;

public class ElementConditionalModifier implements CreativeModifier {
    @Nullable
    private final Set<Integer> presetIdIn;
    @Nullable
    private final Set<Integer> presetIdNotIn;
    private CreativeModifier delegate;

    protected ElementConditionalModifier(@NotNull CreativeModifier delegate,
                                         @Nullable Set<Integer> presetIdIn,
                                         @Nullable Set<Integer> presetIdNotIn) {
        this.delegate = delegate;
        this.presetIdIn = presetIdIn;
        this.presetIdNotIn = presetIdNotIn;
    }

    @NotNull
    public static Builder apply(@NotNull CreativeModifier delegate) {
        Builder builder = new Builder();
        builder.setDelegate(delegate);
        return builder;
    }

    @Override
    public void modify(@NotNull DraftCreative draftCreative) {
        if (presetIdIn != null && !presetIdIn.contains(draftCreative.getPresetId())) {
            return;
        }
        if (presetIdNotIn != null && presetIdNotIn.contains(draftCreative.getPresetId())) {
            return;
        }

        delegate.modify(draftCreative);
    }

    public static class Builder {
        private CreativeModifier delegate;
        private Set<Integer> presetIdIn;
        private Set<Integer> presetIdNotIn;

        private Builder() {
        }

        protected Builder setDelegate(CreativeModifier delegate) {
            this.delegate = delegate;
            return this;
        }

        @NotNull
        public ElementConditionalModifier ifPresetIdIn(Integer... presetIdIn) {
            this.presetIdIn = Stream.of(presetIdIn).collect(Collectors.toSet());
            return this.build();
        }

        @NotNull
        public ElementConditionalModifier ifPresetIdNotIn(Integer... presetIdNotIn) {
            this.presetIdNotIn = Stream.of(presetIdNotIn).collect(Collectors.toSet());
            return this.build();
        }

        @NotNull
        public ElementConditionalModifier build() {
            if (presetIdIn != null && presetIdNotIn != null) {
                throw new IllegalStateException("can't filter upon both presetIdIn and presetIdNotIn condition");
            }
            return new ElementConditionalModifier(delegate, presetIdIn, presetIdNotIn);
        }
    }
}
