package ru.yandex.canvas.service.multitype;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.canvas.service.OnCreativeService;
import ru.yandex.canvas.service.multitype.request.MultiCreativeTypeOperationRequest;
import ru.yandex.canvas.service.multitype.singletype.request.SingleCreativeTypeOperationRequest;

@ParametersAreNonnullByDefault
public abstract class MultiCreativeTypeBaseOperation<T extends MultiCreativeTypeOperationRequest<?>>
        implements MultiCreativeTypeOperation<T> {
    private Map<CreativeType, SingleCreativeTypeOperation> operationsByType;
    private Map<CreativeType, OnCreativeService<?>> serviceForType;

    private final OnCreativeServiceProvider onCreativeServiceProvider;

    public MultiCreativeTypeBaseOperation(OnCreativeServiceProvider onCreativeServiceProvider) {
        this.onCreativeServiceProvider = onCreativeServiceProvider;
    }

    public OnCreativeServiceProvider getOnCreativeServiceProvider() {
        return onCreativeServiceProvider;
    }

    @Override
    public void init(Map<CreativeType, SingleCreativeTypeOperation> operationsByType,
            Map<CreativeType, OnCreativeService<?>> filterByType) {
        if (operationsByType.size() != filterByType.size()) {
            throw new IllegalArgumentException("Operations and services maps sizes should match");
        }
        if (!filterByType.keySet().equals(operationsByType.keySet())) {
            throw new IllegalArgumentException("Operations and services types should match");
        }
        this.serviceForType = filterByType;
        this.operationsByType = operationsByType;
    }

    @Override
    public Map<Long, OnCreativeOperationResult> run(T request) {
        Map<Long, OnCreativeOperationResult> result = new HashMap<>();
        for (CreativeType type : getOperationsByType().keySet()) {
            runOnType(type, request, result);
        }
        for (Long creativeId : request.getCreativeIds()) {
            if (!result.containsKey(creativeId)) {
                result.put(creativeId, OnCreativeOperationResult.notFoundOrNotSupported());
            }
        }
        return result;
    }

    protected void runOnType(CreativeType type, T request, Map<Long, OnCreativeOperationResult> result) {
        List<Long> idsForType = filter(type, request.getCreativeIds());
        if (idsForType.isEmpty()) {
            return;
        }
        SingleCreativeTypeOperationRequest requestForType = request.singleTypeRequest(idsForType);
        getOperationsByType().get(type).run(requestForType, result);
    }

    protected List<Long> filter(CreativeType type, List<Long> creativeIds) {
        return getServiceForType().get(type).filterPresent(creativeIds);
    }

    protected Map<CreativeType, OnCreativeService<?>> getServiceForType() {
        return serviceForType;
    }

    protected Map<CreativeType, SingleCreativeTypeOperation> getOperationsByType() {
        return operationsByType;
    }
}
