package ru.yandex.canvas.service.rtbhost;

import java.util.List;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import ru.yandex.adv.direct.CreativeParameters;
import ru.yandex.canvas.model.CreativeData;
import ru.yandex.canvas.model.Size;

import static ru.yandex.canvas.utils.AdaptiveUtils.getMaxImageSize;

public final class ConstructorData {
    private final List<ConstructorDataElement> elements;
    private final String template;
    private final String backgroundColor;
    private final String borderColor;

    /**
     * Dont's pass into json this keys for non-Adaptive creatives
     */
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final Integer minWidth;
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final Integer minHeight;
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final Integer maxWidth;
    @JsonInclude(JsonInclude.Include.NON_NULL)
    private final Integer maxHeight;

    @JsonInclude(JsonInclude.Include.NON_NULL)
    @JsonProperty("creative_parameters")
    @JsonSerialize(using = CreativeParametersSerializer.class)
    private CreativeParameters creativeParameters;

    /**
     * min size - it's minimal support size for adaptive preset, it's const and define in preset
     * max size - it's minimal plus max size of user upload image
     */
    public ConstructorData(CreativeData creativeData) {
        this.elements = creativeData.toRTBHostExportData();
        this.template = creativeData.getBundle().getName();
        this.backgroundColor = creativeData.getOptions().getBackgroundColor();
        this.borderColor = creativeData.getOptions().getBorderColor();

        if (creativeData.getOptions().getIsAdaptive()) {
            this.minWidth = creativeData.getOptions().getMinWidth();
            this.minHeight = creativeData.getOptions().getMinHeight();
            Integer maxWidth = minWidth;
            Integer maxHeight = minHeight;

            Size maxImageSize = getMaxImageSize(creativeData);
            if (maxImageSize != null) {
                maxWidth += maxImageSize.getWidth();
                maxHeight += maxImageSize.getHeight();
            }
            this.maxWidth = maxWidth;
            this.maxHeight = maxHeight;
        } else {
            this.minWidth = null;
            this.minHeight = null;
            this.maxWidth = null;
            this.maxHeight = null;
        }

        HtmlParametersBuilder builder = new HtmlParametersBuilder();
        builder.setCreativeData(creativeData);
        builder.setMinSize(minWidth, minHeight);
        builder.setMaxSize(maxWidth, maxHeight);

        creativeParameters = CreativeParameters.newBuilder()
                .setHtml(builder.build())
                .build();
    }

    public List<ConstructorDataElement> getElements() {
        return elements;
    }

    public String getTemplate() {
        return template;
    }

    public String getBackgroundColor() {
        return backgroundColor;
    }

    public String getBorderColor() {
        return borderColor;
    }

    public Integer getMinWidth() {
        return minWidth;
    }

    public Integer getMinHeight() {
        return minHeight;
    }

    public Integer getMaxWidth() {
        return maxWidth;
    }

    public Integer getMaxHeight() {
        return maxHeight;
    }

    public static final class ConstructorDataElement {
        private final String type;
        private final String text;
        private final Integer width;
        private final Integer height;
        private final String url;
        private final String color;

        // from Image
        public ConstructorDataElement(String type, String url, Integer width, Integer height) {
            this.url = url;
            this.width = width;
            this.height = height;
            this.type = type;
            this.color = null;
            this.text = null;
        }

        public ConstructorDataElement(String type, String color, String text) {
            this.type = type;
            this.color = color;
            this.text = text;
            this.url = null;
            this.width = null;
            this.height = null;
        }

        public String getType() {
            return type;
        }

        public String getText() {
            return text;
        }

        public Integer getWidth() {
            return width;
        }

        public Integer getHeight() {
            return height;
        }

        public String getUrl() {
            return url;
        }

        public String getColor() {
            return color;
        }
    }
}
