package ru.yandex.canvas.service.rtbhost.helpers;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.List;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import ru.yandex.adv.direct.video.VideoParameters;
import ru.yandex.canvas.TimeDelta;
import ru.yandex.canvas.model.video.addition.AdditionElement;
import ru.yandex.canvas.model.video.addition.options.ButtonElementOptions;
import ru.yandex.canvas.service.RTBHostExportService;
import ru.yandex.canvas.service.rtbhost.RtbHostUploadHelper;
import ru.yandex.canvas.service.video.DCParams;
import ru.yandex.canvas.service.video.PreviewData;
import ru.yandex.canvas.service.video.VAST;
import ru.yandex.canvas.service.video.VideoCreativesService;
import ru.yandex.canvas.service.video.VideoParametersBuilder;
import ru.yandex.direct.bs.dspcreative.model.DspCreativeExportEntry;

import static java.util.Collections.emptyList;
import static ru.yandex.canvas.VideoConstants.VIDEO_MOTION_CREATIVE_ID;
import static ru.yandex.canvas.VideoConstants.VIDEO_MOTION_DURATION;
import static ru.yandex.canvas.VideoConstants.VIDEO_MOTION_SKIP_OFFSET;
import static ru.yandex.canvas.VideoConstants.VIDEO_MOTION_THEME;
import static ru.yandex.canvas.VideoConstants.VIDEO_MOTION_VPAID_PCODE_URL;

public class VideoMotionService implements RtbHostUploadHelper<VideoMotionService.VideoMotion> {

    private RTBHostExportService rtbHostExportService;

    public VideoMotionService(RTBHostExportService rtbHostExportService) {
        this.rtbHostExportService = rtbHostExportService;
    }

    private static VAST.Builder videoMotionVast() {
        return VAST.builder()
                .setVpaidPcodeUrl(VIDEO_MOTION_VPAID_PCODE_URL)
                .setCreativeId(VIDEO_MOTION_CREATIVE_ID)
                .setTheme(VIDEO_MOTION_THEME)
                .setStrmPrefix(null)
                .setHasAbuseButton(true)
                .setHasSkipButton(true)
                .setSkipOffset(new TimeDelta(VIDEO_MOTION_SKIP_OFFSET))
                .setDuration(new TimeDelta(VIDEO_MOTION_DURATION))
                .setMediaFiles(emptyList())
                .setElements(List.of(
                        new AdditionElement(AdditionElement.ElementType.BUTTON)
                                .withAvailable(true)
                                .withOptions(new ButtonElementOptions())
                ))
                .setSoundBtn(0L)
                .setCountdown(1L)
                .setAdLabel(1L)
                .setShowVideoClicks(true)
                .setUseVpaidImpressions(false)
                .setAddPixelImpression(false)
                .setAuctionRenderUrlMacros(true)
                .setUseTrackingEvents(true);
    }

    @Override
    public DspCreativeExportEntry toImportDspCreativeEntry(VideoMotion object,
                                                           ObjectMapper objectMapper) {
        VAST vast;

        try {
            vast = videoMotionVast().build();
        } catch (IOException | URISyntaxException e) {
            throw new RuntimeException(e.getMessage());
        }

        VideoCreativesService.VideoCreativeData videoCreativeData = new VideoCreativesService.VideoCreativeData();
        videoCreativeData.setTemplate(VIDEO_MOTION_THEME);
        videoCreativeData.setUnderlayerID(0);
        videoCreativeData.setDuration(10);
        videoCreativeData.setVideoParameters(videoMotionCreativeParameters());

        try {
            return DspCreativeExportEntry.builder()
                    .setDspId(1L)
                    .setCreativeId(VIDEO_MOTION_CREATIVE_ID)
                    .setCreativeVersionId(VIDEO_MOTION_CREATIVE_ID)
                    .setData(vast.getParameterizedXml())
                    .setConstructorData(objectMapper.writeValueAsString(videoCreativeData))
                    .setEnabled(true)
                    .setVideo(true)
                    .setAudio(false)
                    .setStaticData(String.format("{\"creative_id\":\"%d\"}", VIDEO_MOTION_CREATIVE_ID))
                    .setPostmoderated(true)
                    .setStatic(false)
                    .setTag("yabs")
                    .setWidth(0)
                    .setHeight(0)
                    .build();
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e.getMessage());
        }
    }

    static class VideoMotion {

    }

    public void sendVideoMotiontoRtbHost() {
        rtbHostExportService.exportToRtbHost(Collections.singletonList(new VideoMotion()), this);
    }

    public String getVideoMotionPreview(PreviewData previewData)
            throws URISyntaxException, IOException {
        DCParams dcParams = new DCParams(previewData, null, false);

        return videoMotionVast()
                .setShowVideoClicks(false)
                .build()
                .injectDcParams(dcParams);
    }


    private static VideoParameters videoMotionCreativeParameters() {
        VideoParametersBuilder videoParametersBuilder = new VideoParametersBuilder();
        videoParametersBuilder.setTheme(VIDEO_MOTION_THEME);
        videoParametersBuilder.setCreativeId(VIDEO_MOTION_CREATIVE_ID);
        videoParametersBuilder.setDuration(10.0);
        videoParametersBuilder.setMediaFiles(emptyList());

        return videoParametersBuilder.build();
    }
}
